/*
 * Copyright (c) 2016, Peter Rader. All rights reserved.
 *  ___ ___               __                 ______         __     __  __         __
 * |   |   |.-----..----.|  |_ .-----..----.|   __ \.--.--.|  |--.|  ||__|.-----.|  |--.
 * |   |   ||  -__||  __||   _||  _  ||   _||    __/|  |  ||  _  ||  ||  ||__ --||     |
 *  \_____/ |_____||____||____||_____||__|  |___|   |_____||_____||__||__||_____||__|__|
 *
 * http://www.gnu.org/licenses/gpl-3.0.html
 */
package net.vectorpublish.desktop.vp.article.text;

import java.awt.FontMetrics;
import java.util.Collection;
import java.util.Iterator;

import net.vectorpublish.desktop.vp.Startup;

public class TextChunkCalculator implements Iterator<String> {

	private final float percentage;
	private final float width;
	private String textLeft;
	private final FontMetrics fm;
	private final int calculatedChars;
	private Collection<TextChunkCouncillor> chunkCouncil = Startup.context.getBeansOfType(TextChunkCouncillor.class)
			.values();

	public TextChunkCalculator(float width, String trimmedParagraph, FontMetrics fm) {
		this.width = width;
		this.textLeft = trimmedParagraph;
		this.fm = fm;
		int pixel = fm.stringWidth(trimmedParagraph);
		percentage = width / pixel;
		calculatedChars = (int) (textLeft.length() * percentage);
	}

	@Override
	public boolean hasNext() {
		return !textLeft.isEmpty();
	}

	@Override
	public String next() {
		int calculatedChars = this.calculatedChars;
		while (calculatedChars + 1 <= textLeft.length()
				&& fm.stringWidth(textLeft.substring(0, calculatedChars + 1)) <= width) {
			calculatedChars++;
		}
		while (calculatedChars < textLeft.length() && fm.stringWidth(textLeft.substring(0, calculatedChars)) > width) {
			calculatedChars--;
		}
		calculatedChars--; // remember, the '-' must have its place.

		if (textLeft.length() <= calculatedChars) {
			try {
				return textLeft;
			} finally {
				textLeft = "";
			}
		}

		int splitBeforeIndex = calculatedChars;
		boolean needCheck;
		do {
			needCheck = false;
			for (TextChunkCouncillor textChunkCouncillor : chunkCouncil) {
				if (textChunkCouncillor.hasVeto(textLeft, splitBeforeIndex)) {
					splitBeforeIndex--;
					if (splitBeforeIndex == 0) {
						needCheck = false;
						splitBeforeIndex = 8;
						System.err.println("Councillor " + textChunkCouncillor + " blocks split after "
								+ textLeft.substring(0, splitBeforeIndex + 16) + ".");
						break;
					}
					needCheck = true;
					break;
				}
			}
		} while (needCheck);
		String line = textLeft.substring(0, splitBeforeIndex);
		if (!line.endsWith(" ")) {
			line += "-";
		}
		textLeft = textLeft.substring(line.length() - 1).trim();
		return line;
	}
}
