/*
 * Copyright (c) 2016, Peter Rader. All rights reserved.
 *  ___ ___               __                 ______         __     __  __         __
 * |   |   |.-----..----.|  |_ .-----..----.|   __ \.--.--.|  |--.|  ||__|.-----.|  |--.
 * |   |   ||  -__||  __||   _||  _  ||   _||    __/|  |  ||  _  ||  ||  ||__ --||     |
 *  \_____/ |_____||____||____||_____||__|  |___|   |_____||_____||__||__||_____||__|__|
 *
 * http://www.gnu.org/licenses/gpl-3.0.html
 */
package net.vectorpublish.desktop.vp;

import java.awt.event.ActionEvent;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;

import javax.annotation.PostConstruct;
import javax.inject.Inject;
import javax.inject.Named;

import net.vectorpublish.desktop.vp.ah.SetArticleData;
import net.vectorpublish.desktop.vp.api.history.Redo;
import net.vectorpublish.desktop.vp.api.layer.LayerSelectionListener;
import net.vectorpublish.desktop.vp.api.ui.Dialog;
import net.vectorpublish.desktop.vp.api.ui.ToolBar;
import net.vectorpublish.desktop.vp.api.ui.VPAbstractAction;
import net.vectorpublish.desktop.vp.api.ui.Warn;
import net.vectorpublish.desktop.vp.api.vpd.VectorPublishNode;
import net.vectorpublish.desktop.vp.article.ArticleTexts;
import net.vectorpublish.desktop.vp.article.TextReader;
import net.vectorpublish.desktop.vp.split.SplitNode;
import net.vectorpublish.desktop.vp.ui.ImageKey;
import net.vectorpublish.desktop.vp.utils.SetUtils;

@Named
public class SetArticleButton extends VPAbstractAction implements LayerSelectionListener {

	public SetArticleButton() {
		super(ArticleTexts.SET_ARTICLE, ArticleTexts.SET_ARTICLE_TOOLTIP, true);
	}

	@Inject
	private final ToolBar tb = null;

	@Inject
	private final Dialog dlg = null;

	private SplitNode splitter;

	@Inject
	private final History hist = null;

	@Inject
	private final Redo redo = null;

	@Inject
	private final Warn warn = null;

	@Inject
	private final TextReader[] readers = null;

	@Override
	public void actionPerformed(ActionEvent e) {
		new Thread(new Runnable() {

			@Override
			public void run() {
				List<String> paragraphs = null;
				while (paragraphs == null) {
					File file = dlg.showOpenFile("", "File containing text");
					for (TextReader textReader : readers) {
						try {
							paragraphs = textReader.read(file);
						} catch (IOException exception) {
							warn.addWarning(exception.getMessage());
						}
						if (paragraphs != null) {
							break;
						}
					}
					if (paragraphs == null) {
						Boolean retry;
						try {
							retry = dlg.confirm(ArticleTexts.NS, "No text available", "This is not a vaild text-file.",
									"Change", "Close").get();
						} catch (InterruptedException | ExecutionException e1) {
							return;
						}
						if (retry != null && !retry) {
							return;
						}
					}
				}
				SetArticleData data = new SetArticleData(SetUtils.nodeToImmutableIndex(splitter), paragraphs);
				SetArticleStep step = new SetArticleStep(hist, hist.getCurrentDocument().getLastExecutedHistoryStep(),
						data);
				redo.actionPerformed(e);
			}
		}).start();

	}

	@PostConstruct
	public void setup() {
		setIcons(ArticleTexts.NS, ImageKey.get("article.add"));
		tb.add(this);
	}

	@Override
	public void notify(Set<VectorPublishNode> selection) {
		if (selection.size() != 1) {
			setEnabled(false);
		} else {
			VectorPublishNode selected = selection.iterator().next();
			if (selected instanceof SplitNode) {
				SplitNode splitNode = (SplitNode) selected;
				if (splitNode.getChildCount() < 2) {
					splitter = splitNode;
					setEnabled(true);
				}
			} else {
				setEnabled(false);
			}
		}
	}
}
