package net.sozal.stackwriter.api.domain;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * @author serkan
 */
public final class Frame {

    private final Class clazz;
    private final Method method;
    private final int line;
    private final LocalVariable[] locals;

    public Frame(Class<?> clazz, Method method, int line, LocalVariable[] locals) {
        this.clazz = clazz;
        this.method = method;
        this.line = line;
        this.locals = locals;
    }

    public Class getClazz() {
        return clazz;
    }

    public Method getMethod() {
        return method;
    }

    public int getLine() {
        return line;
    }

    public Map<String, LocalVariable> getLocals() {
        if (locals == null || locals.length == 0) {
            return Collections.emptyMap();
        }

        Map<String, LocalVariable> localsMap = new HashMap<>();
        for (Frame.LocalVariable localVariable : locals) {
            if (localVariable != null) {
                localsMap.put(localVariable.getName(), localVariable);
            }
        }

        return localsMap;
    }

    @Override
    public String toString() {
        return "Frame{"
            + "class=" + clazz.getName()
            + ", method=" + method.getDeclaringClass().getName() + "." + method.getName()
            + ", lineNo=" + line
            + ", locals=" + Arrays.toString(locals)
            + '}';
    }

    public static final class LocalVariable {

        final String name;
        final Object value;
        final Class<?> type;

        public LocalVariable(String name, Object value, char typeSignature) {
            this.name = name;
            this.value = value;
            this.type = getType(value, typeSignature);
        }

        private static Class<?> getType(Object value, char typeSignature) {
            switch(typeSignature) {
                case 'B':
                    return byte.class;
                case 'Z':
                    return boolean.class;
                case 'C':
                    return char.class;
                case 'S':
                    return short.class;
                case 'I':
                    return int.class;
                case 'F':
                    return float.class;
                case 'J':
                    return long.class;
                case 'D':
                    return double.class;
                default:
                    return value != null ? value.getClass() : null;
            }
        }

        public String getName() {
            return name;
        }

        public Object getValue() {
            return value;
        }

        public Class<?> getType() {
            return type;
        }

        @Override
        public String toString() {
            return "LocalVariable{"
                + "name='" + name + '\''
                + ", value=" + value
                + ", type=" + (type != null ? type.getName() : "?")
                + '}';
        }

    }

}
