package net.sozal.stackwriter.agent;

import net.sozal.stackwriter.agent.logger.Logger;
import net.sozal.stackwriter.api.ErrorFilter;
import net.sozal.stackwriter.api.ErrorListener;
import net.sozal.stackwriter.api.domain.Frame;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * @author serkan
 */
public final class ExceptionSnapshotSupport {

    static {
        Logger.debug("<ExceptionSnapshotSupport> Loaded by classloader " +
                ExceptionSnapshotSupport.class.getClassLoader());
    }

    private static final ThreadLocal<WeakHashMap<Throwable, Frame[]>> cache =
        new ThreadLocal<WeakHashMap<Throwable, Frame[]>>() {
            @Override
            protected WeakHashMap<Throwable, Frame[]> initialValue() {
                return new WeakHashMap<>();
            }
        };
    private static final ThreadLocal<Boolean> onErrorInProgress = new ThreadLocal<>();

    private static final Set<String> appPackages = new HashSet<>();
    private static final Set<Class<? extends Throwable>> ignoredExceptionTypes =
            new HashSet<Class<? extends Throwable>>() {{
                add(ClassNotFoundException.class);
            }};
    private static ErrorFilter errorFilter;
    private static final List<ErrorListener> errorListeners = new CopyOnWriteArrayList<>();

    private ExceptionSnapshotSupport() {
    }

    public static boolean shouldTakeSnapshot(Throwable error, int numFrames) {
        if (Boolean.TRUE.equals(onErrorInProgress.get())) {
            return false;
        }
        if (ignoredExceptionTypes.contains(error.getClass())) {
            return false;
        }
        try {
            ErrorFilter ef = errorFilter;

            // Don't take snapshot for all exceptions if there is no filter
            if (appPackages.isEmpty() && ef == null) {
                return false;
            }

            // Many libraries/frameworks seem to rethrow the same object with trimmed stacktraces,
            // which means later ("smaller") throws would overwrite the existing object in cache.
            // For this reason we prefer the throw with the greatest stack length...
            Map<Throwable, Frame[]> weakMap = cache.get();
            Frame[] existing = weakMap.get(error);
            if (existing != null && numFrames <= existing.length) {
                return false;
            }

            if (!appPackages.isEmpty()) {
                boolean inAppPackages = false;

                // Check each frame against all "in app" package prefixes
                for (StackTraceElement stackTraceElement : error.getStackTrace()) {
                    for (String appFrame : appPackages) {
                        if (stackTraceElement.getClassName().startsWith(appFrame)) {
                            inAppPackages = true;
                            break;
                        }
                    }
                }

                if (!inAppPackages) {
                    return false;
                }
            }

            if (ef == null) {
                return true;
            } else {
                return ef.shouldTakeSnapshot(error);
            }
        } catch (Throwable t) {
            Logger.error(
                    "<ExceptionSnapshotSupport> " +
                            "Error occurred while checking whether snapshot should be take on error: " + error.getMessage(),
                    t);
            return false;
        }
    }

    public static void onError(Throwable error, Frame[] frames) {
        if (Boolean.TRUE.equals(onErrorInProgress.get())) {
            return;
        }
        try {
            onErrorInProgress.set(true);
            Map<Throwable, Frame[]> weakMap = cache.get();
            weakMap.put(error, frames);
            for (ErrorListener el: errorListeners) {
                el.onError(error, frames);
            }
        } catch (Throwable t) {
            Logger.error("<ExceptionSnapshotSupport> Error occurred on error: " + error.getMessage(), t);
        } finally {
            onErrorInProgress.set(false);
        }
    }

    public static void addAppPackage(String appPackage) {
        appPackages.add(appPackage);
    }

    public static void addIgnoredExceptionType(Class<? extends Throwable> exceptionType) {
        Logger.debug(String.format(
                "<ExceptionSnapshotSupport> Adding ignored exception type (in classloader=%s): %s",
                exceptionType.getClassLoader(), exceptionType));
        ignoredExceptionTypes.add(exceptionType);
    }

    public static void useErrorFilter(ErrorFilter errorFilter) {
        Logger.debug(String.format(
                "<ExceptionSnapshotSupport> Using error filter (in classloader=%s): %s",
                errorFilter.getClass().getClassLoader(), errorFilter));
        ExceptionSnapshotSupport.errorFilter = errorFilter;
    }

    public static void registerErrorListener(ErrorListener errorListener) {
        Logger.debug(String.format(
                "<ExceptionSnapshotSupport> Registering error listener (in classloader=%s): %s",
                errorListener.getClass().getClassLoader(), errorListener));
        errorListeners.add(errorListener);
    }

}
