/*
 * Decompiled with CFR 0.152.
 */
package org.schwefel.kv;

import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.rocksdb.ColumnFamilyDescriptor;
import org.rocksdb.ColumnFamilyHandle;
import org.rocksdb.ColumnFamilyOptions;
import org.rocksdb.DBOptions;
import org.rocksdb.FlushOptions;
import org.rocksdb.Options;
import org.rocksdb.ReadOptions;
import org.rocksdb.RocksDB;
import org.rocksdb.RocksDBException;
import org.rocksdb.RocksIterator;
import org.rocksdb.Transaction;
import org.rocksdb.TransactionDB;
import org.rocksdb.TransactionDBOptions;
import org.rocksdb.TransactionOptions;
import org.rocksdb.WriteBatch;
import org.rocksdb.WriteOptions;
import org.schwefel.kv.Batch;
import org.schwefel.kv.BatchImpl;
import org.schwefel.kv.ForEachAll;
import org.schwefel.kv.ForEachKeyValue;
import org.schwefel.kv.ForEachRange;
import org.schwefel.kv.Kind;
import org.schwefel.kv.KindImpl;
import org.schwefel.kv.KindManagement;
import org.schwefel.kv.LexicographicByteArrayComparator;
import org.schwefel.kv.MinMaxKeyIt;
import org.schwefel.kv.Stats;
import org.schwefel.kv.StoreException;
import org.schwefel.kv.StoreOps;
import org.schwefel.kv.Transactional;
import org.schwefel.kv.Tx;

public final class KVStore
implements StoreOps,
KindManagement {
    private static final long FLUSH_TIME_WINDOW_MILLIS = 985L;
    private static final long FLUSH_BATCH_SIZE = 20000L;
    private static final Charset UTF8 = Charset.forName("UTF-8");
    private static final Logger logger = Logger.getLogger(KVStore.class.getName());
    private volatile boolean open = false;
    private long totalSinceLastFsync = 0L;
    private long lastSync;
    private TransactionDB txnDb;
    private TransactionDBOptions txnDbOptions;
    private TransactionOptions txnOpts;
    private DBOptions options;
    private ColumnFamilyOptions columnFamilyOptions;
    private WriteOptions writeOptions;
    private ReadOptions readOptions;
    private FlushOptions flushOptions;
    private FlushOptions flushOptionsNoWait;
    private final HashMap<String, KindImpl> kinds = new HashMap();
    private final String path;
    private final Stats stats = new Stats();

    public KVStore(Path dir) {
        this.path = (String)KVStore.wrapEx(() -> Objects.requireNonNull(dir).toFile().getCanonicalPath());
        KVStore.wrapEx(() -> Files.createDirectories(dir, new FileAttribute[0]));
        this.open();
    }

    private void open() {
        this.options = new DBOptions();
        this.options.setCreateIfMissing(true);
        this.options.setErrorIfExists(false);
        this.options.setKeepLogFileNum(2L);
        this.options.setDeleteObsoleteFilesPeriodMicros(3600000000L);
        this.options.setIncreaseParallelism(Math.max(Runtime.getRuntime().availableProcessors(), 2));
        this.columnFamilyOptions = new ColumnFamilyOptions();
        this.writeOptions = new WriteOptions();
        this.readOptions = new ReadOptions();
        this.flushOptions = new FlushOptions();
        this.flushOptions.setWaitForFlush(true);
        this.flushOptionsNoWait = new FlushOptions();
        this.flushOptionsNoWait.setWaitForFlush(false);
        this.txnDbOptions = new TransactionDBOptions();
        this.txnDb = (TransactionDB)KVStore.wrapEx(() -> this.openDatabase());
        this.txnOpts = new TransactionOptions();
        this.open = true;
        this.lastSync = System.currentTimeMillis();
    }

    private TransactionDB openDatabase() throws RocksDBException {
        try (Options opts = new Options(this.options, this.columnFamilyOptions);){
            List<byte[]> families = RocksDB.listColumnFamilies(opts, this.path);
            ArrayList<ColumnFamilyDescriptor> cfDescs = new ArrayList<ColumnFamilyDescriptor>();
            for (byte[] cfName : families) {
                cfDescs.add(new ColumnFamilyDescriptor(cfName, this.columnFamilyOptions));
            }
            if (cfDescs.isEmpty()) {
                cfDescs.add(new ColumnFamilyDescriptor(RocksDB.DEFAULT_COLUMN_FAMILY, this.columnFamilyOptions));
            }
            ArrayList<ColumnFamilyHandle> cfHandles = new ArrayList<ColumnFamilyHandle>(cfDescs.size());
            TransactionDB txnDb = TransactionDB.open(this.options, this.txnDbOptions, this.path, cfDescs, cfHandles);
            for (ColumnFamilyHandle handle : cfHandles) {
                KindImpl kind = new KindImpl(handle.getName(), handle);
                this.kinds.put(kind.name(), kind);
            }
            TransactionDB transactionDB = txnDb;
            return transactionDB;
        }
    }

    @Override
    public synchronized void close() {
        if (!this.isOpen()) {
            return;
        }
        this.open = false;
        KVStore.ignoreEx(() -> this.syncWAL());
        KVStore.ignoreEx(() -> this.flush());
        this.closeCfHandles();
        this.kinds.clear();
        KVStore.close(this.txnDb);
        KVStore.close(this.txnDbOptions);
        KVStore.close(this.txnOpts);
        KVStore.close(this.columnFamilyOptions);
        KVStore.close(this.writeOptions);
        KVStore.close(this.readOptions);
        KVStore.close(this.flushOptions);
        KVStore.close(this.flushOptionsNoWait);
        KVStore.close(this.options);
        this.txnDb = null;
        this.txnDbOptions = null;
        this.txnOpts = null;
        this.columnFamilyOptions = null;
        this.writeOptions = null;
        this.readOptions = null;
        this.flushOptions = null;
        this.flushOptionsNoWait = null;
        this.options = null;
    }

    private void closeCfHandles() {
        for (KindImpl kind : this.kinds.values()) {
            KVStore.close(kind.handle());
        }
    }

    @Override
    public KindManagement getKindManagement() {
        this.validateOpen();
        return this;
    }

    @Override
    public synchronized Set<Kind> getKinds() {
        this.validateOpen();
        return new TreeSet<Kind>(this.kinds.values());
    }

    @Override
    public synchronized Kind getKind(String kindName) {
        this.validateOpen();
        return this.kinds.get(kindName);
    }

    @Override
    public synchronized Kind getOrCreateKind(String kindName) {
        if (Objects.requireNonNull(kindName).isEmpty()) {
            throw new IllegalArgumentException("kindName: ");
        }
        this.validateOpen();
        Kind kind = this.getKind(kindName);
        if (kind == null) {
            kind = (Kind)KVStore.wrapEx(() -> this.createKind(kindName));
        }
        return kind;
    }

    private Kind createKind(String kindName) throws RocksDBException {
        ColumnFamilyHandle handle = this.txnDb.createColumnFamily(new ColumnFamilyDescriptor(kindName.getBytes(UTF8), this.columnFamilyOptions));
        KindImpl kind = new KindImpl(handle.getName(), handle);
        this.kinds.put(kind.name(), kind);
        return kind;
    }

    @Override
    public synchronized Kind getDefaultKind() {
        this.validateOpen();
        return this.getKind("default");
    }

    @Override
    public synchronized void compact(String kindName) {
        if (Objects.requireNonNull(kindName).isEmpty()) {
            throw new IllegalArgumentException("kindName: ");
        }
        this.validateOpen();
        Kind kind = this.getKind(kindName);
        if (kind != null) {
            this.compact_((KindImpl)kind);
        }
    }

    @Override
    public synchronized void compactAll() {
        this.validateOpen();
        for (KindImpl kind : this.kinds.values()) {
            this.compact_(kind);
        }
    }

    private void compact_(KindImpl kind) {
        try {
            this.txnDb.compactRange(kind.handle());
        }
        catch (RocksDBException e) {
            throw new StoreException(e);
        }
    }

    @Override
    public synchronized void put(Kind kind, byte[] key, byte[] value) {
        long start = System.nanoTime();
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(key, "key cannot be null");
        this.validateOpen();
        try {
            this.put_(kind, key, value);
        }
        catch (RocksDBException e) {
            throw new StoreException(e);
        }
        finally {
            this.stats.allOpsTimeNanos.accept(System.nanoTime() - start);
        }
    }

    private void put_(Kind kind, byte[] key, byte[] value) throws RocksDBException {
        long putStart = System.nanoTime();
        try (Transaction txn = this.txnDb.beginTransaction(this.writeOptions, this.txnOpts);){
            txn.put(((KindImpl)kind).handle(), key, value);
            txn.commit();
            this.stats.putTimeNanos.accept(System.nanoTime() - putStart);
            this.occasionalWalSync();
        }
    }

    private void occasionalWalSync() {
        ++this.totalSinceLastFsync;
        if (System.currentTimeMillis() - this.lastSync >= 985L) {
            this.syncWAL();
            this.lastSync = System.currentTimeMillis();
            this.totalSinceLastFsync = 0L;
        } else if (this.totalSinceLastFsync % 20000L == 0L) {
            this.syncWAL();
            this.lastSync = System.currentTimeMillis();
            this.totalSinceLastFsync = 0L;
        }
    }

    @Override
    public synchronized void putIfAbsent(Kind kind, byte[] key, byte[] value) {
        long start = System.nanoTime();
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(key, "key cannot be null");
        this.validateOpen();
        try {
            if (this.get_(kind, key) == null) {
                this.put_(kind, key, value);
            }
        }
        catch (RocksDBException e) {
            throw new StoreException(e);
        }
        finally {
            this.stats.allOpsTimeNanos.accept(System.nanoTime() - start);
        }
    }

    @Override
    public synchronized byte[] deleteIfPresent(Kind kind, byte[] key) {
        long start = System.nanoTime();
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(key, "key cannot be null");
        this.validateOpen();
        byte[] oldVal = null;
        try {
            oldVal = this.get_(kind, key);
            if (oldVal != null) {
                this.delete_(kind, key);
            }
        }
        catch (RocksDBException e) {
            throw new StoreException(e);
        }
        finally {
            this.stats.allOpsTimeNanos.accept(System.nanoTime() - start);
        }
        return oldVal;
    }

    @Override
    public synchronized byte[] singleDeleteIfPresent(Kind kind, byte[] key) {
        long start = System.nanoTime();
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(key, "key cannot be null");
        this.validateOpen();
        byte[] oldVal = null;
        try {
            oldVal = this.get_(kind, key);
            if (oldVal != null) {
                this.singleDelete_(kind, key);
            }
        }
        catch (RocksDBException e) {
            throw new StoreException(e);
        }
        finally {
            this.stats.allOpsTimeNanos.accept(System.nanoTime() - start);
        }
        return oldVal;
    }

    @Override
    public synchronized byte[] updateIfPresent(Kind kind, byte[] key, byte[] value) {
        long start = System.nanoTime();
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(key, "key cannot be null");
        this.validateOpen();
        byte[] oldVal = null;
        try {
            oldVal = this.get_(kind, key);
            if (oldVal != null) {
                this.put_(kind, key, value);
            }
        }
        catch (RocksDBException e) {
            throw new StoreException(e);
        }
        finally {
            this.stats.allOpsTimeNanos.accept(System.nanoTime() - start);
        }
        return oldVal;
    }

    @Override
    public synchronized byte[] get(Kind kind, byte[] key) {
        long start = System.nanoTime();
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(key, "key cannot be null");
        this.validateOpen();
        try {
            byte[] byArray = this.get_(kind, key);
            return byArray;
        }
        catch (RocksDBException e) {
            throw new StoreException(e);
        }
        finally {
            this.stats.allOpsTimeNanos.accept(System.nanoTime() - start);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private byte[] get_(Kind kind, byte[] key) throws RocksDBException {
        long start = System.nanoTime();
        try {
            byte[] byArray = this.txnDb.get(((KindImpl)kind).handle(), this.readOptions, key);
            return byArray;
        }
        finally {
            this.stats.getTimeNanos.accept(System.nanoTime() - start);
        }
    }

    @Override
    public synchronized void delete(Kind kind, byte[] key) {
        long start = System.nanoTime();
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(key, "key cannot be null");
        this.validateOpen();
        try {
            this.delete_(kind, key);
        }
        catch (RocksDBException e) {
            throw new StoreException(e);
        }
        finally {
            this.stats.allOpsTimeNanos.accept(System.nanoTime() - start);
        }
    }

    private void delete_(Kind kind, byte[] key) throws RocksDBException {
        long delStart = System.nanoTime();
        try (Transaction txn = this.txnDb.beginTransaction(this.writeOptions, this.txnOpts);){
            txn.delete(((KindImpl)kind).handle(), key);
            txn.commit();
            this.stats.deleteTimeNanos.accept(System.nanoTime() - delStart);
            this.occasionalWalSync();
        }
    }

    @Override
    public synchronized void singleDelete(Kind kind, byte[] key) {
        long start = System.nanoTime();
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(key, "key cannot be null");
        this.validateOpen();
        try {
            this.singleDelete_(kind, key);
        }
        catch (RocksDBException e) {
            throw new StoreException(e);
        }
        finally {
            this.stats.allOpsTimeNanos.accept(System.nanoTime() - start);
        }
    }

    private void singleDelete_(Kind kind, byte[] key) throws RocksDBException {
        long delStart = System.nanoTime();
        try (Transaction txn = this.txnDb.beginTransaction(this.writeOptions, this.txnOpts);){
            txn.singleDelete(((KindImpl)kind).handle(), key);
            txn.commit();
            this.stats.deleteTimeNanos.accept(System.nanoTime() - delStart);
            this.occasionalWalSync();
        }
    }

    @Override
    public synchronized void writeBatch(Batch batch) {
        long start = System.nanoTime();
        Objects.requireNonNull(batch, "batch cannot be null");
        this.validateOpen();
        WriteBatch wb = ((BatchImpl)batch).cedeOwnership();
        if (wb != null) {
            try {
                this.txnDb.write(this.writeOptions, wb);
            }
            catch (RocksDBException e) {
                throw new StoreException(e);
            }
            finally {
                KVStore.close(wb);
                long delta = System.nanoTime() - start;
                this.stats.allOpsTimeNanos.accept(delta);
                this.stats.batchTimeNanos.accept(delta);
                this.occasionalWalSync();
            }
        }
    }

    @Override
    public synchronized void syncWAL() {
        if (this.isOpen()) {
            long start = System.nanoTime();
            try {
                if (this.txnDb.isOwningHandle()) {
                    this.txnDb.flushWal(true);
                }
            }
            catch (RocksDBException e) {
                throw new StoreException(e);
            }
            finally {
                long delta = System.nanoTime() - start;
                this.stats.allOpsTimeNanos.accept(delta);
                this.stats.walTimeNanos.accept(delta);
            }
        }
    }

    @Override
    public boolean isOpen() {
        return this.open;
    }

    private void validateOpen() {
        if (!this.isOpen()) {
            throw new StoreException("KVStore " + this.path + " is closed");
        }
    }

    @Override
    public synchronized void flush() {
        this.flush_(this.flushOptions);
    }

    @Override
    public synchronized void flushNoWait() {
        this.flush_(this.flushOptionsNoWait);
    }

    private void flush_(FlushOptions flushOptions) {
        if (this.isOpen()) {
            long start = System.nanoTime();
            try {
                this.txnDb.flush(flushOptions);
            }
            catch (RocksDBException e) {
                throw new StoreException(e);
            }
            finally {
                long delta = System.nanoTime() - start;
                this.stats.allOpsTimeNanos.accept(delta);
                this.stats.flushTimeNanos.accept(delta);
            }
        }
    }

    @Override
    public synchronized ForEachKeyValue scanAll(Kind kind) {
        Objects.requireNonNull(kind, "kind cannot be null");
        this.validateOpen();
        RocksIterator it = Objects.requireNonNull(this.txnDb.newIterator(((KindImpl)kind).handle()));
        this.stats.incOpenCursorsCount();
        it.seekToFirst();
        return new ForEachAll(it, this.stats, this);
    }

    @Override
    public synchronized ForEachKeyValue scanAll(Kind kind, byte[] beginKey) {
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(beginKey, "beginKey cannot be null");
        this.validateOpen();
        RocksIterator it = Objects.requireNonNull(this.txnDb.newIterator(((KindImpl)kind).handle()));
        this.stats.incOpenCursorsCount();
        it.seek(beginKey);
        return new ForEachAll(it, this.stats, this);
    }

    @Override
    public synchronized ForEachKeyValue scanRange(Kind kind, byte[] beginKey, byte[] endKey) {
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(beginKey, "beginKey cannot be null");
        Objects.requireNonNull(endKey, "endKey cannot be null");
        this.validateOpen();
        RocksIterator it = Objects.requireNonNull(this.txnDb.newIterator(((KindImpl)kind).handle()));
        this.stats.incOpenCursorsCount();
        it.seek(beginKey);
        return new ForEachRange(it, endKey, this.stats, this);
    }

    @Override
    public synchronized byte[] findMinKey(Kind kind) {
        Objects.requireNonNull(kind, "kind cannot be null");
        this.validateOpen();
        RocksIterator it = Objects.requireNonNull(this.txnDb.newIterator(((KindImpl)kind).handle()));
        this.stats.incOpenCursorsCount();
        return MinMaxKeyIt.findMinKey(it, this.stats);
    }

    @Override
    public synchronized byte[] findMinKeyByPrefix(Kind kind, byte[] keyPrefix) {
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(keyPrefix, "keyPrefix cannot be null");
        this.validateOpen();
        RocksIterator it = Objects.requireNonNull(this.txnDb.newIterator(((KindImpl)kind).handle()));
        this.stats.incOpenCursorsCount();
        return MinMaxKeyIt.findMinKey(it, this.stats, keyPrefix);
    }

    @Override
    public synchronized byte[] findMaxKey(Kind kind) {
        Objects.requireNonNull(kind, "kind cannot be null");
        this.validateOpen();
        RocksIterator it = Objects.requireNonNull(this.txnDb.newIterator(((KindImpl)kind).handle()));
        this.stats.incOpenCursorsCount();
        return MinMaxKeyIt.findMaxKey(it, this.stats);
    }

    @Override
    public synchronized byte[] findMaxKeyByPrefix(Kind kind, byte[] keyPrefix) {
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(keyPrefix, "keyPrefix cannot be null");
        this.validateOpen();
        RocksIterator it = Objects.requireNonNull(this.txnDb.newIterator(((KindImpl)kind).handle()));
        this.stats.incOpenCursorsCount();
        return MinMaxKeyIt.findMaxKey(it, this.stats, keyPrefix);
    }

    @Override
    public synchronized byte[] findMinKeyByLowerBound(Kind kind, byte[] lowerBound) {
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(lowerBound, "lowerBound cannot be null");
        this.validateOpen();
        return MinMaxKeyIt.findMinKeyByLowerBound(this.txnDb, ((KindImpl)kind).handle(), this.stats, lowerBound);
    }

    @Override
    public synchronized byte[] findMaxKeyByUpperBound(Kind kind, byte[] upperBound) {
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(upperBound, "upperBound cannot be null");
        this.validateOpen();
        return MinMaxKeyIt.findMaxKeyByUpperBound(this.txnDb, ((KindImpl)kind).handle(), this.stats, upperBound);
    }

    @Override
    public synchronized byte[] findMaxKeyLessThan(Kind kind, byte[] keyPrefix, byte[] upperBound) {
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(keyPrefix, "keyPrefix cannot be null");
        Objects.requireNonNull(upperBound, "upperBound cannot be null");
        this.validateOpen();
        if (keyPrefix.length >= upperBound.length && LexicographicByteArrayComparator.lexicographicalCompare(keyPrefix, upperBound) > 0) {
            return null;
        }
        RocksIterator it = Objects.requireNonNull(this.txnDb.newIterator(((KindImpl)kind).handle()));
        this.stats.incOpenCursorsCount();
        return MinMaxKeyIt.findMaxKeyLessThan(it, this.stats, keyPrefix, upperBound);
    }

    @Override
    public synchronized byte[] findMinKeyGreaterThan(Kind kind, byte[] keyPrefix, byte[] lowerBound) {
        Objects.requireNonNull(kind, "kind cannot be null");
        Objects.requireNonNull(keyPrefix, "keyPrefix cannot be null");
        Objects.requireNonNull(lowerBound, "lowerBound cannot be null");
        this.validateOpen();
        if (keyPrefix.length >= lowerBound.length && LexicographicByteArrayComparator.lexicographicalCompare(keyPrefix, lowerBound) < 0) {
            return null;
        }
        RocksIterator it = Objects.requireNonNull(this.txnDb.newIterator(((KindImpl)kind).handle()));
        this.stats.incOpenCursorsCount();
        return MinMaxKeyIt.findMinKeyGreaterThan(it, this.stats, keyPrefix, lowerBound);
    }

    @Override
    public Batch createBatch() {
        return new BatchImpl();
    }

    @Override
    public synchronized Tx startTx() {
        return new Transactional(this.txnDb.beginTransaction(this.writeOptions, this.txnOpts), this.readOptions, this.getStats());
    }

    @Override
    public synchronized Stats getStats() {
        return this.stats;
    }

    private static void close(AutoCloseable ac) {
        if (ac != null) {
            try {
                ac.close();
            }
            catch (Exception ignore) {
                logger.log(Level.INFO, "", ignore);
            }
        }
    }

    private static Object wrapEx(ThrowingSupplier block) {
        try {
            return block.get();
        }
        catch (Exception e) {
            logger.log(Level.WARNING, "", e);
            throw new StoreException(e);
        }
    }

    private static void ignoreEx(Runnable block) {
        try {
            block.run();
        }
        catch (Exception ignore) {
            logger.log(Level.INFO, "", ignore);
        }
    }

    private static interface ThrowingSupplier {
        public Object get() throws Exception;
    }
}

