/*
 * Copyright 2008-2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javarequirementstracer.systemtest;

import java.io.File;
import java.io.IOException;

import javarequirementstracer.AttributeId;
import javarequirementstracer.FileUtils;
import javarequirementstracer.JavaRequirementsTracer;
import javarequirementstracer.ReporterUtils;
import javarequirementstracer.Requirements;

import org.junit.Assert;
import org.junit.Test;


/**
 * @author Ronald Koster
 */
public class ReporterTest {
	
	static final String MAIN_PARAMS_FILENAME = "../jrt-reporter/src/main/config/traceability_params.properties";
	static final String SYSTEM_TEST_PARAMS_FILENAME = "src/test/config/reporter_system_test_traceability_params.properties";
	static final String MAIN_REPORT_FILENAME = "../jrt-reporter/target/traceability.html";

	
	@Test
	@Requirements("UC-Generate-Report")
	public void ucGenerateReport() throws IOException {
		final String content = createReport(MAIN_PARAMS_FILENAME);

		final int indexCodeCoverage = getIndexCodeCoverage(content);
		final int indexRequirementsCoverage = getIndexRequirementsCoverage(content);
		final int indexSettings = getIndexSettings(content);
		
		assertPresent(content, indexCodeCoverage, indexRequirementsCoverage, ReporterUtils.formatPercentage("100", "00"));
		assertPresent(content, indexRequirementsCoverage, indexSettings, ReporterUtils.formatPercentage("100", "00"));
	}

	@Test
	@Requirements("UC-Parseable-Report")
	public void ucParseableReport() throws IOException {
		final String content = createReport(SYSTEM_TEST_PARAMS_FILENAME);
		for (AttributeId att : AttributeId.values()) {
			assertPresent(content, 0, content.length(), "id='" + att.toString() + "'");
		}
	}

	@Test
	@Requirements("UC-Standalone-Report")
	public void ucStandaloneReport() throws IOException {
		String paramsFilenames = MAIN_PARAMS_FILENAME;
		JavaRequirementsTracer.main(new String[]{"-params.filenames=" + paramsFilenames,
				"-build.number=" + STHelper.getBuildNumber()});
		File file = new File(MAIN_REPORT_FILENAME);
		Assert.assertTrue(MAIN_REPORT_FILENAME, file.exists());
		Assert.assertTrue("Report must be new.", System.currentTimeMillis() >= file.lastModified());
	}
	
	private void assertPresent(String content, int index1, int index2, String... strings) {
		FileUtils.assertPresent(content, index1, index2, strings);
	}
	
	private static int getIndexCodeCoverage(String str) {
		return str.indexOf("CodeCoverage");
	}
	
	private static int getIndexRequirementsCoverage(String str) {
		return str.indexOf("RequirementsCoverage");
	}
	
	private static int getIndexSettings(String str) {
		return str.indexOf("Settings");
	}
	
	private String createReport(String paramsFilename) throws IOException {
		return STHelper.createReport(paramsFilename);
	}
}