/*
 * Copyright 2008-2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javarequirementstracer.systemtest;

import java.io.IOException;

import javarequirementstracer.FileUtils;
import javarequirementstracer.ReportAggregator;
import javarequirementstracer.ReporterUtils;
import javarequirementstracer.Requirements;

import org.junit.Test;


/**
 * @author Ronald Koster
 */
public class AggregatorTest {

	private static final String REPORTER_MAIN_PARAMS_FILENAME = ReporterTest.MAIN_PARAMS_FILENAME;
	private static final String REPORTER_SYSTEM_TEST_PARAMS_FILENAME = ReporterTest.SYSTEM_TEST_PARAMS_FILENAME;
	
	private static final String PLUGIN_MAIN_PARAMS_FILENAME = "../jrt-maven-plugin/src/main/config/traceability_params.properties";
	private static final String PLUGIN_SYSTEM_TEST_PARAMS_FILENAME = "src/test/config/plugin_system_test_traceability_params.properties";
	
	private static final String OVERVIEW_MAIN_PARAMS_FILENAME = "src/test/config/traceability_overview_params_main.properties";
	private static final String OVERVIEW_MAIN_REPORT_FILENAME = "target/traceability_overview_main.html";
	private static final String OVERVIEW_SYSTEM_TEST_PARAMS_FILENAME = "src/test/config/traceability_overview_params_system_test.properties";
	private static final String OVERVIEW_SYSTEM_TEST_REPORT_FILENAME = "target/traceability_overview_system_test.html";

	
	@Test
	@Requirements("UC-Overview-Report")
	public void mainCodeUcOverviewReport() throws IOException {
		STHelper.createReport(REPORTER_MAIN_PARAMS_FILENAME);
		STHelper.createReport(PLUGIN_MAIN_PARAMS_FILENAME);
		ReportAggregator agg = new ReportAggregator();
		agg.setOverviewParamsFilename(OVERVIEW_MAIN_PARAMS_FILENAME);
		agg.run();
		String report = FileUtils.readFileAsString(OVERVIEW_MAIN_REPORT_FILENAME);
		doAsserts(report, "Main Code", "57", "14");
	}
	
	@Test
	@Requirements("UC-Overview-Report")
	public void systemTestCodeUcOverviewReport() throws IOException {
		STHelper.createReport(REPORTER_SYSTEM_TEST_PARAMS_FILENAME);
		STHelper.createReport(PLUGIN_SYSTEM_TEST_PARAMS_FILENAME);
		ReportAggregator agg = new ReportAggregator();
		agg.setOverviewParamsFilename(OVERVIEW_SYSTEM_TEST_PARAMS_FILENAME);
		agg.run();
		String report = FileUtils.readFileAsString(OVERVIEW_SYSTEM_TEST_REPORT_FILENAME);
		doAsserts(report, "System Test Code", "42", "86");
	}
	
	private void doAsserts(String report, String suffix, String integerPart, String fraction) {
		int index = FileUtils.assertPresent(report, 0, -1, "Traceability Overview for JavaRequirementsTracer " + suffix);
		index = FileUtils.assertPresent(report, index, -1, "Reporter " + suffix);
		double dWitdh = Double.parseDouble(integerPart) + Double.parseDouble(fraction) / 100D;
		long iWidth = Math.round(dWitdh * 1.5D);
		index = FileUtils.assertPresent(report, index, -1, iWidth + "px");
		FileUtils.assertPresent(report, index, -1, ReporterUtils.formatPercentage(integerPart, fraction));
	}
}