/*
 * Copyright 2008-2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javarequirementstracer;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.HashSet;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.core.type.filter.AnnotationTypeFilter;
import org.springframework.core.type.filter.AssignableTypeFilter;
import org.springframework.util.ClassUtils;

/**
 * Implementation of {@link JavaRequirementsTracer#scanUntraceable()}.
 * 
 * @author Ronald Koster
 */
final class UntraceableScanner extends AbstractScanner {
	
	UntraceableScanner(TraceProperties properties) {
		super(properties);
	}
	
	SortedSet<String> run() {
		final SortedSet<String> typeNames = new TreeSet<String>();
		for (Class<?> cl : getTypes(getProperties().getIncludePackageNames())) {
			if (exclude(cl)) {
				continue;
			}
			int modifiers = cl.getModifiers();
			if (Modifier.isPublic(modifiers) || Modifier.isProtected(modifiers)) {
				String typeName = getProperties().getShortTypeName(cl.getName());
				typeNames.add(typeName);
			}
		}
		return typeNames;
	}
	
	private Set<Class<?>> getTypes(final Set<String> packageNames) {
		final Set<Class<?>> list = new HashSet<Class<?>>();
		
		// Scan for all classess in the classpath with the Requirements annotation.
		final ClassPathScanner scanner = new ClassPathScanner(false);
		setResourceLoader(scanner);
		scanner.addIncludeFilter(new AssignableTypeFilter(Object.class));
		scanner.addExcludeFilter(new AnnotationTypeFilter(Requirements.class));
		scanner.addExcludeFilter(new AnnotationTypeFilter(SuppressTraceabilityWarnings.class));

		// Search the given packages.
		for (String basePackageName : packageNames) {
			final Set<BeanDefinition> components = scanner.findCandidateComponents(basePackageName);
			for (BeanDefinition component : components) {
				String className = component.getBeanClassName();
				Class<?> cl = ClassUtils.resolveClassName(className, getClassLoader());
				if (!isAnnotationInherited(Requirements.class, cl)) {
					list.add(cl);
				}
			}
		}
		return list;
	}
	
	private boolean isAnnotationInherited(Class<? extends Annotation> annotationType, Class<?> clazz) {
		Class<?> supercl = clazz.getSuperclass();
		if (supercl != null && isAnnotationInherited(annotationType, new Class<?>[]{supercl})) {
			return true;
		}
		Class<?>[] interfs = clazz.getInterfaces();
		if (isAnnotationInherited(annotationType, interfs)) {
			return true;
		}
		return false;
	}		
	
	private boolean isAnnotationInherited(Class<? extends Annotation> annotationType, Class<?>[] superTypes) {
		for (Class<?> superType : superTypes) {
			if (AnnotationUtils.findAnnotation(superType, annotationType) != null) {
				return true;
			}
			Method[] meths = superType.getMethods();
			for (Method meth : meths) {
				if (AnnotationUtils.findAnnotation(meth, annotationType) != null) {
					return true;
				}
			}
		}
		return false;
	}
}