/*
 * Copyright 2008-2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javarequirementstracer;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;

import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.core.type.filter.AnnotationTypeFilter;
import org.springframework.util.ClassUtils;

/**
 * Implementation of {@link JavaRequirementsTracer#scanRequirements()}.
 * 
 * @author Ronald Koster
 */
final class RequirementsScanner extends AbstractScanner {
	
	static final String METHOD_SEPARATOR = "#";
	
	RequirementsScanner(TraceProperties properties) {
		super(properties);
	}
	
	SortedMap<String, SortedSet<String>> run() {
		final SortedMap<String, SortedSet<String>> codeNamesTo = new TreeMap<String, SortedSet<String>>();
		for (Class<?> cl : getTypes(getProperties().getIncludePackageNames())) {
			if (exclude(cl)) {
				continue;
			}
			// NB. Requirements annotations cannot be inherited. (No @Inherited)
			if (cl.isAnnotationPresent(Requirements.class)) {
				Requirements reqs = cl.getAnnotation(Requirements.class);
				addLabels(codeNamesTo, cl, reqs);
			}
			for (Constructor<?> con : cl.getDeclaredConstructors()) {
				if (con.isAnnotationPresent(Requirements.class)) {
					Requirements reqs = con.getAnnotation(Requirements.class);
					addLabels(codeNamesTo, con, reqs);
				}
			}
			for (Method meth : cl.getDeclaredMethods()) {
				if (meth.isAnnotationPresent(Requirements.class)) {
					Requirements reqs = meth.getAnnotation(Requirements.class);
					addLabels(codeNamesTo, meth, reqs);
				}
			}
		}
		return codeNamesTo;
	}
	
	private Set<Class<?>> getTypes(final Set<String> packageNames) {
		final Set<Class<?>> list = new HashSet<Class<?>>();
		
		// Scan for all classess in the classpath with the Requirements annotation.
		final ClassPathScanner scanner = new ClassPathScanner(false);
		setResourceLoader(scanner);
		scanner.addIncludeFilter(new AnnotationTypeFilter(Requirements.class));

		// Search the given packages.
		for (String basePackageName : packageNames) {
			final Set<BeanDefinition> components = scanner.findCandidateComponents(basePackageName);
			for (BeanDefinition component : components) {
				String className = component.getBeanClassName();
				list.add(ClassUtils.resolveClassName(className, getClassLoader()));
			}
		}
		return list;
	}
	
	private <T> void addLabels(final Map<String, SortedSet<String>> codeNamesToLabels, final T type,
			final Requirements reqs) {
		final Set<String> labels = getLabels(codeNamesToLabels, type);
		labels.addAll(Arrays.asList(reqs.value()));
	}
	
	private <T> Set<String> getLabels(final Map<String, SortedSet<String>> codeNamesToLabels, final T type) {
		String typeName = getProperties().getShortTypeName(getTypeName(type));
		SortedSet<String> set = codeNamesToLabels.get(typeName);
		if (set == null) {
			set = new TreeSet<String>();
			codeNamesToLabels.put(typeName, set);
		}
		return set;
	}
	
	private String getTypeName(final Object type) {
		if (type instanceof Class) {
			Class<?> cl = (Class<?>) type;
			return cl.getName();
		} else if (type instanceof Constructor) {
			Constructor<?> con = (Constructor<?>) type;
			String name = con.getName();
			int index = name.lastIndexOf('.'); 
			return name + METHOD_SEPARATOR + name.substring(index + 1);
		} else if (type instanceof Method) {
			Method meth = (Method) type;
			return meth.getDeclaringClass().getName() + METHOD_SEPARATOR + meth.getName();
		}
		throw new IllegalArgumentException("type=" + type); 
	}
}