/*
 * Copyright 2008-2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javarequirementstracer;

import java.util.SortedMap;
import java.util.SortedSet;

/**
 * Class for generating a traceablity report. It scans for {@link Requirements} annotations.
 * @see #init() 
 * 
 * @author Ronald Koster
 */
@Requirements("UC-Generate-Report")
public final class JavaRequirementsTracerBean {
	
	@SuppressWarnings("unused")
	private static final Logger LOGGER = Logger.getInstance(JavaRequirementsTracerBean.class);
	
	private String paramsFilename;
	private String buildNumber;
	private ClassLoader classLoader;
	
	private final TraceProperties properties = new TraceProperties();
	

	/**
	 * @return See {@link #init()}.
	 */
	public String getParamsFilename() {
		return this.paramsFilename;
	}

	/**
	 * Injector, see {@link #init()}.
	 */
	public void setParamsFilename(final String paramsFilename) {
		this.paramsFilename = paramsFilename;
	}

	/**
	 * @return See {@link #init()}.
	 */
	public String getExpectedLabelsFilename() {
		return this.properties.getExpectedLabelsFilename();
	}

	/**
	 * @return See {@link #init()}.
	 */
	public String getReportFilename() {
		return this.properties.getReportFilename();
	}

	/**
	 * Injector. ClassLoader to be used for loading this class and the classes to be scanned.
	 * Default (null) system ClassLoader is used.
	 * Needed for javarequirementstracer.mojo.JrtMojo mainly/only.
	 */
	public void setClassLoader(final ClassLoader classLoader) {
		this.classLoader = classLoader;
	}

	/**
	 * Injector. Optional buildnumber to be used for the reporting.
	 * @param buildNumber the buildnumber.
	 */
	public void setBuildNumber(final String buildNumber) {
		this.buildNumber = buildNumber;
	}

	/**
	 * Prepare a scan for {@link JavaRequirementsTracer#printUsage() current options}.
	 * @throws UncheckedException Wrapper around {@link java.io.FileNotFoundException} {@link java.io.IOException}, 
	 * in case loading the properties from {@link #setParamsFilename(String)} or
	 * {@link #setExpectedLabelsFilename(String)} throws such an exception.  
	 */
	public void init() {
		this.properties.loadParams(this.paramsFilename);
		this.properties.setBuildNumber(this.buildNumber);
		this.properties.setClassLoader(this.classLoader);
	}
	
	/**
	 * Create a report using {@link JavaRequirementsTracer#printUsage() current options}.
	 * 
	 * @throws UncheckedException Idem as at {@link #init()}.  
	 */
	public void run() {
		init();
		SortedMap<String, SortedSet<String>> codeNamesToLabels = scanRequirements();
		SortedSet<String> untraceableTypes = scanUntraceable();
		write(codeNamesToLabels, untraceableTypes);
	}

	/**
	 * Does actual tracing (ie. scanning for {@link Requirements} annotations). 
	 * @return A map mapping codeElementNames (aka. codeNames) to the requirements labels
	 * (aka. labels) they implement according to their annotations.
	 * @throws IllegalStateException If {@link #init()} has not been invoked yet. 
	 */
	public SortedMap<String, SortedSet<String>> scanRequirements() {
		validate();
		final RequirementsScanner scanner = new RequirementsScanner(this.properties);
		return scanner.run();
	}
	
	/**
	 * @return All public or protected types (classes/interfaces) not annotated with
	 * {@link Requirements}, nor with {@link SuppressTraceabilityWarnings},  
	 * nor having a supertype annotated with {@link Requirements}. 
	 * NB. Having a supertype annotated with {@link SuppressTraceabilityWarnings} will
	 * not exclude it from the untraceable report. This is because many Use Case specific service
	 * implementations may implement a general technical (thus annotated with 
	 * {@link SuppressTraceabilityWarnings}) interface.
	 * @throws IllegalStateException Idem as at {@link #scanRequirements()}. 
	 */
	public SortedSet<String> scanUntraceable() {
		validate();
		final UntraceableScanner scanner = new UntraceableScanner(this.properties);
		return scanner.run();
	}
	
	/**
	 * Write scanned traceability map to the report file indicated by {@link #setReportFileName(String)}.
	 * Also reports:
	 * <ul>
	 * <li/>Which requirement labels were not found. 
	 * <li/>All unknown requirement labels found (usually typos). 
	 * <li/>All untraceable public or protected types (classes/interfaces), as retuned 
	 * by {@link #scanUntraceable()}. (usually the to do list)
	 * </ul>
	 * @param codeNamesToLabels As returned by {@link #scanRequirements()}.
	 * @param untraceableTypes As retuned by {@link #scanUntraceable()}.
	 * @throws RuntimeException Wrapper around {@link java.io.IOException}, in case writing to the report file
	 * throws such an exception.  
	 */
	public void write(final SortedMap<String, SortedSet<String>> codeNamesToLabels,
			final SortedSet<String> untraceableTypes) {
		TraceReporter reporter = new TraceReporter(this.properties, codeNamesToLabels, untraceableTypes);
		reporter.run();
	}
	
	private void validate() {
		if (this.properties == null) {
			throw new IllegalStateException("Not initialized yet.");
		}
	}
}