/*
 * Copyright 2008-2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javarequirementstracer;

import java.io.File;
import java.io.IOException;

import junit.framework.Assert;

import org.junit.Test;



/**
 * @author Ronald Koster
 */
@Requirements({"UC20"})
public class JavaRequirementsTracerTest {
	private static final String PARAMS_FILENAMES_KEY = "params.filenames";
	
	@Requirements({"UC20.9"})
	@Test
	public void usage() {
		final String reportFilename = "target/ut_traceability.html";(new JavaRequirementsTracerBean()).getReportFilename();
		
		// Test usage is printed, and report is not generated. 
		final File file = new File(reportFilename);
		long lastModified = -1;
		if (file.exists()) {
			lastModified = file.lastModified();
		}
		JavaRequirementsTracer.main(new String[]{"-h"});
		if (file.exists()) {
			Assert.assertEquals("File should not have been touched.", lastModified, file.lastModified());
		}
	}
	
	@Test
	public void invalidArgs() {
		Exception ex = null;
		try {
			JavaRequirementsTracer.main(new String[]{"-" + PARAMS_FILENAMES_KEY + "="});
		} catch (IllegalArgumentException iaex) {
			ex = iaex;
		}
		Assert.assertNotNull(ex);
	}
	
	/**
	 * This method generates a complete report. Can be used as an example.
	 */
	@Test
	public void main() throws IOException {
		final String buildNr = "1234321";
		JavaRequirementsTracerBean tracer = new JavaRequirementsTracerBean();
		tracer.setParamsFilename("src/test/config/ut_traceability_params.properties");
		tracer.setBuildNumber(buildNr);
		tracer.run();
		
		final String content = loadFile(tracer.getReportFilename());
		
		// Default test code is excluded.
		assertNotPresent(content, 0, -1, "TestTracer");
		assertNotPresent(content, 0, -1, "UC20.9");
		
		final int indexSummary = content.indexOf("Summary");
		final int indexSettings = content.indexOf("Settings", indexSummary);
		final int indexExcludePackageNames = content.indexOf("excludePackageNames:", indexSettings);
		final int indexExcludeTypeNames = content.indexOf("excludeTypeNames:", indexExcludePackageNames);
		final int indexMissing = content.indexOf("Missing Requirements", indexExcludeTypeNames);
		final int indexUnknown = content.indexOf("Unknown Requirements", indexMissing);
		final int indexUntraceable = content.indexOf("Untraceable Types", indexMissing);
		final int indexReqsToCode = content.indexOf("Requirement &rarr; Code Elements", indexUnknown);
		final int indexCodeToReqs = content.indexOf("Code Element &rarr; Requirements", indexReqsToCode);
		final int indexDescriptions = content.indexOf("Descriptions", indexCodeToReqs);
		Assert.assertTrue(0 < indexSummary);
		Assert.assertTrue(indexSummary < indexSettings);
		Assert.assertTrue(indexSettings < indexExcludePackageNames);
		Assert.assertTrue(indexExcludePackageNames < indexExcludeTypeNames);
		Assert.assertTrue(indexExcludeTypeNames < indexMissing);
		Assert.assertTrue(indexMissing < indexUnknown);
		Assert.assertTrue(indexUnknown < indexUntraceable);
		Assert.assertTrue(indexUntraceable < indexReqsToCode);
		Assert.assertTrue(indexReqsToCode < indexCodeToReqs);
		Assert.assertTrue(indexCodeToReqs < indexDescriptions);
		
		// Summary section.
		assertPresent(content, indexSummary, indexSettings,
				"Timestamp", buildNr, "CodeCoverage", ReporterUtils.formatPercentage("77", "78"), "= traceableTypeCount/allTypesCount = 7/9");
		assertPresent(content, indexSummary, indexSettings,
				"RequirementsCoverage", ReporterUtils.formatPercentage("50", "00"), "= (foundLabelCount - unknownLabelCount)/requiredLabelCount = 5/10");
		assertPresent(content, indexSummary, indexSettings,
				"requiredLabelCount = 10 =? missingLabelCount + foundLabelCount - unknownLabelCount = 5 + 13 - 8 = 10&nbsp;&nbsp;&nbsp;&nbsp;...OK"); 
		assertPresent(content, indexSummary, indexSettings,
				"allTypesCount = 9 =? traceableTypeCount + untraceableTypeCount = 7 + 2 = 9&nbsp;&nbsp;&nbsp;&nbsp;...OK");
		
		// Settings section.
		final String rootPackName = "javarequirementstracer";
		assertPresent(content, indexSummary, indexMissing, rootPackName);
		assertNotPresent(content, indexMissing, -1, rootPackName);
		
		final String excludePackageNames = "[javarequirementstracer.exclude]";
		assertPresent(content, indexExcludePackageNames, indexMissing, excludePackageNames);
		assertNotPresent(content, indexMissing, -1, excludePackageNames);
		
		final String excludeTypeNames = "[javarequirementstracer.DummyExcl, javarequirementstracer.IDummyExcl]";
		assertPresent(content, indexExcludeTypeNames, indexMissing, excludeTypeNames);
		assertNotPresent(content, indexMissing, -1, excludeTypeNames);

		// Missing reqs.
		assertPresent(content, indexMissing, indexUnknown, "count = 5", "UC10");
		assertNotPresent(content, indexMissing, indexUnknown, "UC10.1", "UC11.1");
		
		// Unknown reqs.
		assertPresent(content, indexUnknown, indexReqsToCode, "count = 8", "UC11.1");
		assertNotPresent(content, indexUnknown, indexReqsToCode, "UC10");
		
		// Untraceable.
		assertPresent(content, indexUntraceable, indexReqsToCode, "count = 2", "Dummy5");
		assertNotPresent(content, indexUntraceable, indexReqsToCode, "Tracer", "Dummy6", "Dummy7",
				"Dummy1Impl", "Dummy1ImplExt", "Dummy3Sub", "Dummy3SubImpl", "DummyExcl", "IDummyExcl");
		
		// Reqs to Code section.
		assertPresent(content, indexReqsToCode, indexCodeToReqs, "count = 13", "Dummy1", "Dummy2", "Dummy3", "UC10.1");
		
		// Code to Reqs section.
		assertPresent(content, indexCodeToReqs, indexDescriptions, "count = 12", "Dummy1", "Dummy2", "Dummy3", "UC11.1");
		
		// Descriptions.
		assertPresent(content, indexDescriptions, -1, "count = 10", "UC10", "UC20");
	}
	
	private String loadFile(String filename) throws IOException {
		return FileUtils.readFileAsString(filename);
	}
	
	private void assertPresent(String content, int index1, int index2, String... strings) {
		FileUtils.assertPresent(content, index1, index2, strings);
	}
	
	private void assertNotPresent(String content, int index1, int index2, String... strings) {
		FileUtils.assertNotPresent(content, index1, index2, strings);
	}
}