/*
 * Copyright 2008-2009 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package javarequirementstracer;


/**
 * Class for generating a traceablity report by running as a standalone Java application.
 * It scans for {@link Requirements} annotations.
 * @see #printUsage() 
 * 
 * @author Ronald Koster
 */
@Requirements("UC-Standalone-Report")
public final class JavaRequirementsTracer {
	
	private static final Logger LOGGER = Logger.getInstance(JavaRequirementsTracer.class);
	
	public static final String DEFAULT_PARAMS_FILENAMES =
		"src/main/config/traceability_params.properties, src/test/config/traceability_params.properties";

	private static final String PARAMS_FILENAMES_KEY = "params.filenames";
	private static final String BUILD_NUMBER_KEY = "build.number";
	
	private static final String USAGE = "Usage: java " + JavaRequirementsTracer.class.getName() + " {-<optionName>=<value>}"
			+ "\n\nWith <optionName>: "
			+ "\n    " + PARAMS_FILENAMES_KEY + ": Comma separated list of properties filenames each containing"
			+ "\n                     all tuning parameters for a report. For each a report will generated."
			+ "\n                     Default: " + DEFAULT_PARAMS_FILENAMES + "."
			+ "\n                     See also the comments in the properties file itself."
			+ "\n    " + BUILD_NUMBER_KEY + ": Build number of code to be traced. Default: none.\n";


	private JavaRequirementsTracer() {
		// Main class, is also a util class.
	}
	
	/**
	 * Through this method one can create traceability reports by running this application as a standalone
	 * Java application. It scans all class files in the classpath for annotations. See {@link #USAGE}.
	 * <p/>
	 * Alternatively one could create a test class (which is not really a test class) like this:
	 * <pre>
	 * public class TraceabilityReportTest {
	 * 
	 *     &#064;Test
	 *     public void run() {
	 *         JavaRequirementsTracerBean tracer = new JavaRequirementsTracerBean();
	 *
	 *         // Override defaults where needed.
	 *         //setParamsFilename("../Requirements/XxxTraceability_params.properties");
	 *         setBuildNumber(getBuildNumber());
	 *
	 *         tracer.run();
	 *     }
	 *     
	 *     private String getBuildNumber() {
	 *         //Some code to fetch the build number from somewhere. 
	 *     }
	 * }
	 * </pre>
	 * 
	 * @param args See {@link #printUsage()}.
	 */
	public static void main(final String[] args) {
		if (args.length == 1 && (args[0].equals("-h") || args[0].equals("-help") || args[0].equals("--help"))) {
			System.out.println(USAGE);
			return;
		}
		
		String[] paramsFilenames = split(getOption(args, PARAMS_FILENAMES_KEY, DEFAULT_PARAMS_FILENAMES));
		JavaRequirementsTracerBean tracer = new JavaRequirementsTracerBean();
		tracer.setBuildNumber(getOption(args, BUILD_NUMBER_KEY, null));
		for (String paramsFilename : paramsFilenames) {
			tracer.setParamsFilename(paramsFilename);
			tracer.run();
		}
	}
	
	private static String getOption(String[] args, String key, String defaultValue) {
		final String theKey = "-" + key;
		if (args != null) {
			for (String arg : args) {
				if (arg.startsWith(theKey)) {
					LOGGER.debug("arg=" + arg);
					final int index = theKey.length() + 1;
					if (arg.length() <= index) {
						throw new IllegalArgumentException("Invalid value for option: " + key);
					}
					return arg.substring(index, arg.length());
				}
			}
		}
		return defaultValue;
	}
	
	public static String[] split(String str) {
		return str.trim().split(TraceProperties.SEPARATOR);
	}
}