package org.obolibrary.macro;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.obolibrary.obo2owl.Obo2OWLConstants;
import org.obolibrary.obo2owl.Obo2OWLConstants.Obo2OWLVocabulary;
import org.semanticweb.owlapi.model.IRI;
import org.semanticweb.owlapi.model.OWLAnnotation;
import org.semanticweb.owlapi.model.OWLAnnotationAssertionAxiom;
import org.semanticweb.owlapi.model.OWLAnnotationProperty;
import org.semanticweb.owlapi.model.OWLAnnotationPropertyDomainAxiom;
import org.semanticweb.owlapi.model.OWLAnnotationPropertyRangeAxiom;
import org.semanticweb.owlapi.model.OWLAnnotationValue;
import org.semanticweb.owlapi.model.OWLAsymmetricObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLAxiom;
import org.semanticweb.owlapi.model.OWLAxiomVisitorEx;
import org.semanticweb.owlapi.model.OWLClass;
import org.semanticweb.owlapi.model.OWLClassAssertionAxiom;
import org.semanticweb.owlapi.model.OWLClassExpression;
import org.semanticweb.owlapi.model.OWLClassExpressionVisitorEx;
import org.semanticweb.owlapi.model.OWLDataAllValuesFrom;
import org.semanticweb.owlapi.model.OWLDataComplementOf;
import org.semanticweb.owlapi.model.OWLDataExactCardinality;
import org.semanticweb.owlapi.model.OWLDataFactory;
import org.semanticweb.owlapi.model.OWLDataHasValue;
import org.semanticweb.owlapi.model.OWLDataIntersectionOf;
import org.semanticweb.owlapi.model.OWLDataMaxCardinality;
import org.semanticweb.owlapi.model.OWLDataMinCardinality;
import org.semanticweb.owlapi.model.OWLDataOneOf;
import org.semanticweb.owlapi.model.OWLDataPropertyAssertionAxiom;
import org.semanticweb.owlapi.model.OWLDataPropertyDomainAxiom;
import org.semanticweb.owlapi.model.OWLDataPropertyRangeAxiom;
import org.semanticweb.owlapi.model.OWLDataRange;
import org.semanticweb.owlapi.model.OWLDataSomeValuesFrom;
import org.semanticweb.owlapi.model.OWLDataUnionOf;
import org.semanticweb.owlapi.model.OWLDataVisitorEx;
import org.semanticweb.owlapi.model.OWLDatatype;
import org.semanticweb.owlapi.model.OWLDatatypeDefinitionAxiom;
import org.semanticweb.owlapi.model.OWLDatatypeRestriction;
import org.semanticweb.owlapi.model.OWLDeclarationAxiom;
import org.semanticweb.owlapi.model.OWLDifferentIndividualsAxiom;
import org.semanticweb.owlapi.model.OWLDisjointClassesAxiom;
import org.semanticweb.owlapi.model.OWLDisjointDataPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLDisjointObjectPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLDisjointUnionAxiom;
import org.semanticweb.owlapi.model.OWLEquivalentClassesAxiom;
import org.semanticweb.owlapi.model.OWLEquivalentDataPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLEquivalentObjectPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLFacetRestriction;
import org.semanticweb.owlapi.model.OWLFunctionalDataPropertyAxiom;
import org.semanticweb.owlapi.model.OWLFunctionalObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLHasKeyAxiom;
import org.semanticweb.owlapi.model.OWLIndividual;
import org.semanticweb.owlapi.model.OWLInverseFunctionalObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLInverseObjectPropertiesAxiom;
import org.semanticweb.owlapi.model.OWLIrreflexiveObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLLiteral;
import org.semanticweb.owlapi.model.OWLNegativeDataPropertyAssertionAxiom;
import org.semanticweb.owlapi.model.OWLNegativeObjectPropertyAssertionAxiom;
import org.semanticweb.owlapi.model.OWLObjectAllValuesFrom;
import org.semanticweb.owlapi.model.OWLObjectComplementOf;
import org.semanticweb.owlapi.model.OWLObjectExactCardinality;
import org.semanticweb.owlapi.model.OWLObjectHasSelf;
import org.semanticweb.owlapi.model.OWLObjectHasValue;
import org.semanticweb.owlapi.model.OWLObjectIntersectionOf;
import org.semanticweb.owlapi.model.OWLObjectMaxCardinality;
import org.semanticweb.owlapi.model.OWLObjectMinCardinality;
import org.semanticweb.owlapi.model.OWLObjectOneOf;
import org.semanticweb.owlapi.model.OWLObjectProperty;
import org.semanticweb.owlapi.model.OWLObjectPropertyAssertionAxiom;
import org.semanticweb.owlapi.model.OWLObjectPropertyDomainAxiom;
import org.semanticweb.owlapi.model.OWLObjectPropertyExpression;
import org.semanticweb.owlapi.model.OWLObjectPropertyRangeAxiom;
import org.semanticweb.owlapi.model.OWLObjectSomeValuesFrom;
import org.semanticweb.owlapi.model.OWLObjectUnionOf;
import org.semanticweb.owlapi.model.OWLOntology;
import org.semanticweb.owlapi.model.OWLReflexiveObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLSameIndividualAxiom;
import org.semanticweb.owlapi.model.OWLSubAnnotationPropertyOfAxiom;
import org.semanticweb.owlapi.model.OWLSubClassOfAxiom;
import org.semanticweb.owlapi.model.OWLSubDataPropertyOfAxiom;
import org.semanticweb.owlapi.model.OWLSubObjectPropertyOfAxiom;
import org.semanticweb.owlapi.model.OWLSubPropertyChainOfAxiom;
import org.semanticweb.owlapi.model.OWLSymmetricObjectPropertyAxiom;
import org.semanticweb.owlapi.model.OWLTransitiveObjectPropertyAxiom;
import org.semanticweb.owlapi.model.SWRLRule;

/**
 * Empty abstract visitor for macro expansion. This class allows to minimize the
 * code in the actual visitors, as they only need to overwrite the relevant
 * methods.
 */
public abstract class AbstractMacroExpansionVisitor implements
        OWLClassExpressionVisitorEx<OWLClassExpression>,
        OWLDataVisitorEx<OWLDataRange>, OWLAxiomVisitorEx<OWLAxiom> {

    static final Logger LOG = Logger
            .getLogger(AbstractMacroExpansionVisitor.class.getName());
    static public final Set<OWLAnnotation> EMPTY_ANNOTATIONS = Collections.emptySet();

    final OWLDataFactory dataFactory;
    final Map<IRI, String> expandAssertionToMap;
    final Map<IRI, String> expandExpressionMap;

    public OWLAnnotationProperty getOIO_ISEXPANSION() {
        return OIO_ISEXPANSION;
    }

    final protected OWLAnnotationProperty OIO_ISEXPANSION;

    public OWLAnnotation getExpansionMarkerAnnotation() {
        return expansionMarkerAnnotation;
    }

    final protected OWLAnnotation expansionMarkerAnnotation;
    private boolean shouldAddExpansionMarker=false;

    @SuppressWarnings("null")
    protected AbstractMacroExpansionVisitor(OWLOntology ontology, boolean shouldAddExpansionMarker) {
        this(ontology);
        this.shouldAddExpansionMarker = shouldAddExpansionMarker;
    }
    protected AbstractMacroExpansionVisitor(OWLOntology inputOntology) {
        dataFactory = inputOntology.getOWLOntologyManager().getOWLDataFactory();
        expandExpressionMap = new HashMap<IRI, String>();
        expandAssertionToMap = new HashMap<IRI, String>();
        OWLAnnotationProperty expandExpressionAP = dataFactory
                .getOWLAnnotationProperty(Obo2OWLVocabulary.IRI_IAO_0000424
                        .getIRI());
        OWLAnnotationProperty expandAssertionAP = dataFactory
                .getOWLAnnotationProperty(Obo2OWLVocabulary.IRI_IAO_0000425
                        .getIRI());
        OIO_ISEXPANSION=dataFactory.getOWLAnnotationProperty(IRI.create(Obo2OWLConstants.OIOVOCAB_IRI_PREFIX,"is_expansion"));
        expansionMarkerAnnotation = dataFactory.getOWLAnnotation(OIO_ISEXPANSION, dataFactory.getOWLLiteral(true));
        for (OWLObjectProperty p : inputOntology
                .getObjectPropertiesInSignature()) {
            for (OWLOntology o : inputOntology.getImportsClosure()) {
                for (OWLAnnotation a : p.getAnnotations(o, expandExpressionAP)) {
                    OWLAnnotationValue v = a.getValue();
                    if (v instanceof OWLLiteral) {
                        String str = ((OWLLiteral) v).getLiteral();
                        LOG.log(Level.INFO, "mapping {} to {}" + p + str);
                        expandExpressionMap.put(p.getIRI(), str);
                    }
                }
            }
        }
        for (OWLAnnotationProperty p : inputOntology
                .getAnnotationPropertiesInSignature()) {
            for (OWLOntology o : inputOntology.getImportsClosure()) {
                for (OWLAnnotation a : p.getAnnotations(o, expandAssertionAP)) {
                    OWLAnnotationValue v = a.getValue();
                    if (v instanceof OWLLiteral) {
                        String str = ((OWLLiteral) v).getLiteral();
                        LOG.log(Level.INFO, "assertion mapping {} to {}" + p
                                + str);
                        expandAssertionToMap.put(p.getIRI(), str);
                    }
                }
            }
        }
    }

    @Override
    public OWLClassExpression visit(OWLObjectIntersectionOf ce) {
        Set<OWLClassExpression> ops = new HashSet<OWLClassExpression>();
        for (OWLClassExpression op : ce.getOperands()) {
            ops.add(op.accept(this));
        }
        return dataFactory.getOWLObjectIntersectionOf(ops);
    }

    @Override
    public OWLClassExpression visit(OWLObjectUnionOf ce) {
        Set<OWLClassExpression> ops = new HashSet<OWLClassExpression>();
        for (OWLClassExpression op : ce.getOperands()) {
            ops.add(op.accept(this));
        }
        return dataFactory.getOWLObjectUnionOf(ops);
    }

    @Override
    public OWLClassExpression visit(OWLObjectComplementOf ce) {
        return dataFactory.getOWLObjectComplementOf(ce.getOperand()
                .accept(this));
    }

    @Override
    public OWLClassExpression visit(OWLObjectSomeValuesFrom ce) {
        OWLClassExpression filler = ce.getFiller();
        OWLObjectPropertyExpression p = ce.getProperty();
        OWLClassExpression result = null;
        if (p instanceof OWLObjectProperty) {
            result = expandOWLObjSomeVal(filler, p);
        }
        if (result == null) {
            result = dataFactory.getOWLObjectSomeValuesFrom(ce.getProperty(),
                    filler.accept(this));
        }
        return result;
    }

    protected abstract OWLClassExpression expandOWLObjSomeVal(
            OWLClassExpression filler, OWLObjectPropertyExpression p);

    @Override
    public OWLClassExpression visit(OWLObjectHasValue ce) {
        OWLClassExpression result = null;
        OWLIndividual filler = ce.getFiller();
        OWLObjectPropertyExpression p = ce.getProperty();
        if (p instanceof OWLObjectProperty) {
            result = expandOWLObjHasVal(ce, filler, p);
        }
        if (result == null) {
            result = dataFactory.getOWLObjectHasValue(ce.getProperty(), filler);
        }
        return result;
    }

    protected abstract OWLClassExpression expandOWLObjHasVal(
            OWLObjectHasValue desc, OWLIndividual filler,
            OWLObjectPropertyExpression p);

    @Override
    public OWLClassExpression visit(OWLObjectAllValuesFrom ce) {
        return ce.getFiller().accept(this);
    }

    @Override
    public OWLClassExpression visit(OWLObjectMinCardinality ce) {
        OWLClassExpression filler = ce.getFiller().accept(this);
        return dataFactory.getOWLObjectMinCardinality(ce.getCardinality(),
                ce.getProperty(), filler);
    }

    @Override
    public OWLClassExpression visit(OWLObjectExactCardinality ce) {
        return ce.asIntersectionOfMinMax().accept(this);
    }

    @Override
    public OWLClassExpression visit(OWLObjectMaxCardinality ce) {
        OWLClassExpression filler = ce.getFiller().accept(this);
        return dataFactory.getOWLObjectMaxCardinality(ce.getCardinality(),
                ce.getProperty(), filler);
    }

    @Override
    public OWLClassExpression visit(OWLDataSomeValuesFrom ce) {
        OWLDataRange filler = ce.getFiller().accept(this);
        return dataFactory.getOWLDataSomeValuesFrom(ce.getProperty(), filler);
    }

    @Override
    public OWLClassExpression visit(OWLDataAllValuesFrom ce) {
        OWLDataRange filler = ce.getFiller().accept(this);
        return dataFactory.getOWLDataAllValuesFrom(ce.getProperty(), filler);
    }

    @Override
    public OWLClassExpression visit(OWLDataHasValue ce) {
        return ce.asSomeValuesFrom().accept(this);
    }

    @Override
    public OWLClassExpression visit(OWLDataExactCardinality ce) {
        return ce.asIntersectionOfMinMax().accept(this);
    }

    @Override
    public OWLClassExpression visit(OWLDataMaxCardinality ce) {
        int card = ce.getCardinality();
        OWLDataRange filler = ce.getFiller().accept(this);
        return dataFactory.getOWLDataMaxCardinality(card, ce.getProperty(),
                filler);
    }

    @Override
    public OWLClassExpression visit(OWLDataMinCardinality ce) {
        int card = ce.getCardinality();
        OWLDataRange filler = ce.getFiller().accept(this);
        return dataFactory.getOWLDataMinCardinality(card, ce.getProperty(),
                filler);
    }

    @Override
    public OWLDataRange visit(OWLDataOneOf node) {
        // Encode as a data union of and return result
        Set<OWLDataOneOf> oneOfs = new HashSet<OWLDataOneOf>();
        for (OWLLiteral lit : node.getValues()) {
            oneOfs.add(dataFactory.getOWLDataOneOf(lit));
        }
        return dataFactory.getOWLDataUnionOf(oneOfs).accept(this);
    }

    @Override
    public OWLDataRange visit(OWLDataIntersectionOf node) {
        Set<OWLDataRange> ops = new HashSet<OWLDataRange>();
        for (OWLDataRange op : node.getOperands()) {
            ops.add(op.accept(this));
        }
        return dataFactory.getOWLDataIntersectionOf(ops);
    }

    @Override
    public OWLDataRange visit(OWLDataUnionOf node) {
        Set<OWLDataRange> ops = new HashSet<OWLDataRange>();
        for (OWLDataRange op : node.getOperands()) {
            ops.add(op.accept(this));
        }
        return dataFactory.getOWLDataUnionOf(ops);
    }

    // Conversion of non-class expressions to MacroExpansionVisitor
    @Override
    public OWLAxiom visit(OWLSubClassOfAxiom axiom) {
        OWLClassExpression subClass = axiom.getSubClass();
        OWLClassExpression newSubclass = subClass.accept(this);
        OWLClassExpression superClass = axiom.getSuperClass();
        OWLClassExpression newSuperclass = superClass.accept(this);
        if (subClass.equals(newSubclass) && superClass.equals(newSuperclass)) {
            return axiom;
        } else {
            return dataFactory.getOWLSubClassOfAxiom(newSubclass, newSuperclass, getAnnotationsWithOptionalExpansionMarker(axiom));
        }
    }

    public Set<OWLAnnotation> getAnnotationsWithOptionalExpansionMarker(OWLAxiom axiom) {
        if (shouldAddExpansionMarker) {
            Set<OWLAnnotation> annotations = new HashSet<OWLAnnotation>(axiom.getAnnotations());
            annotations.add(expansionMarkerAnnotation);
            return annotations;
        } else {
            return axiom.getAnnotations();
        }
    }

    @Override
    public OWLAxiom visit(OWLDisjointClassesAxiom axiom) {
        Set<OWLClassExpression> ops = new HashSet<OWLClassExpression>();
        boolean sawChange = false;
        for (OWLClassExpression op : axiom.getClassExpressions()) {
            OWLClassExpression newOp = op.accept(this);
            ops.add(newOp);
            if (!op.equals(newOp)) {
                sawChange = true;
            }
        }
        if (sawChange) {
            return dataFactory.getOWLDisjointClassesAxiom(ops, getAnnotationsWithOptionalExpansionMarker(axiom));
        } else {
            return axiom;
        }
    }

    @Override
    public OWLAxiom visit(OWLDisjointUnionAxiom axiom) {
        Set<OWLClassExpression> newOps = new HashSet<OWLClassExpression>();
        boolean sawChange = false;
        for (OWLClassExpression op : axiom.getClassExpressions()) {
            OWLClassExpression newOp = op.accept(this);
            newOps.add(newOp);
            if (!op.equals(newOp)) {
                sawChange = true;
            }
        }
        if (!sawChange) {
            return axiom;
        }
        return dataFactory.getOWLDisjointUnionAxiom(axiom.getOWLClass(), newOps, getAnnotationsWithOptionalExpansionMarker(axiom));
    }


    @Override
    public OWLAxiom visit(OWLDataPropertyDomainAxiom axiom) {
        OWLClassExpression domain = axiom.getDomain();
        OWLClassExpression newDomain = domain.accept(this);
        if (domain.equals(newDomain)) {
            return axiom;
        } else {
            return dataFactory.getOWLDataPropertyDomainAxiom(axiom.getProperty(),
                    newDomain, getAnnotationsWithOptionalExpansionMarker(axiom));
        }
    }

    @Override
    public OWLAxiom visit(OWLObjectPropertyDomainAxiom axiom) {
        OWLClassExpression domain = axiom.getDomain();
        OWLClassExpression newDomain = domain.accept(this);
        if (domain.equals(newDomain)) {
            return axiom;
        } else {
            return dataFactory.getOWLObjectPropertyDomainAxiom(axiom.getProperty(),
                    newDomain, getAnnotationsWithOptionalExpansionMarker(axiom));
        }
    }

    @Override
    public OWLAxiom visit(OWLObjectPropertyRangeAxiom axiom) {
        OWLClassExpression range = axiom.getRange();
        OWLClassExpression newRange = range.accept(this);
        if (range.equals(newRange)) {
            return axiom;
        } else {
            return dataFactory.getOWLObjectPropertyRangeAxiom(axiom.getProperty(),
                    newRange, getAnnotationsWithOptionalExpansionMarker(axiom));
        }
    }

    @Override
    public OWLAxiom visit(OWLDataPropertyRangeAxiom axiom) {
        OWLDataRange range = axiom.getRange();
        OWLDataRange newRange = range.accept(this);
        if (range.equals(newRange)) {
            return axiom;
        } else {
            return dataFactory.getOWLDataPropertyRangeAxiom(axiom.getProperty(),
                    newRange, getAnnotationsWithOptionalExpansionMarker(axiom));
        }
    }

    @Override
    public OWLAxiom visit(OWLClassAssertionAxiom axiom) {
        OWLClassExpression classExpression = axiom.getClassExpression();
        if (classExpression.isAnonymous()) {
            OWLClassExpression newClassExpression = classExpression.accept(this);
            if (!classExpression.equals(newClassExpression)) {
                return dataFactory.getOWLClassAssertionAxiom(newClassExpression, axiom.getIndividual(), getAnnotationsWithOptionalExpansionMarker(axiom));
            }
        }
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLEquivalentClassesAxiom axiom) {
        Set<OWLClassExpression> newExpressions = new HashSet<OWLClassExpression>();
        boolean sawChange = false;
        for (OWLClassExpression expression : axiom.getClassExpressions()) {
            OWLClassExpression newExpression = expression.accept(this);
            newExpressions.add(newExpression);
            if (!expression.equals(newExpression)) {
                sawChange = true;
            }
        }
        if (sawChange) {
            return dataFactory.getOWLEquivalentClassesAxiom(newExpressions, getAnnotationsWithOptionalExpansionMarker(axiom));
        } else {
            return axiom;
        }
    }

    @Override
    public OWLClassExpression visit(OWLClass ce) {
        return ce;
    }

    @Override
    public OWLClassExpression visit(OWLObjectHasSelf ce) {
        return ce;
    }

    @Override
    public OWLClassExpression visit(OWLObjectOneOf ce) {
        return ce;
    }

    @Override
    public OWLDataRange visit(OWLDatatype node) {
        return node;
    }

    @Override
    public OWLDataRange visit(OWLDataComplementOf node) {
        return node;
    }

    @Override
    public OWLAxiom visit(OWLHasKeyAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLDataRange visit(OWLDatatypeRestriction node) {
        return node;
    }

    @Override
    public OWLAxiom visit(OWLNegativeObjectPropertyAssertionAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLAsymmetricObjectPropertyAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLReflexiveObjectPropertyAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLEquivalentObjectPropertiesAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLNegativeDataPropertyAssertionAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLDifferentIndividualsAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLDisjointDataPropertiesAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLDisjointObjectPropertiesAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLObjectPropertyAssertionAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLFunctionalObjectPropertyAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLSubObjectPropertyOfAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLDeclarationAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLAnnotationAssertionAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLSymmetricObjectPropertyAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLFunctionalDataPropertyAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLEquivalentDataPropertiesAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLDataPropertyAssertionAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLTransitiveObjectPropertyAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLIrreflexiveObjectPropertyAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLSubDataPropertyOfAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLInverseFunctionalObjectPropertyAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLSameIndividualAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLSubPropertyChainOfAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLInverseObjectPropertiesAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(SWRLRule rule) {
        return rule;
    }

    @Override
    public OWLAxiom visit(OWLAnnotationPropertyDomainAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLAnnotationPropertyRangeAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLSubAnnotationPropertyOfAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLAxiom visit(OWLDatatypeDefinitionAxiom axiom) {
        return axiom;
    }

    @Override
    public OWLDataRange visit(OWLLiteral node) {
        return null;
    }

    @Override
    public OWLDataRange visit(OWLFacetRestriction node) {
        return null;
    }
}
