/**
 * Copyright Openmind http://www.openmindonline.it
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package it.openutils.magnoliastripes;

import info.magnolia.cms.beans.config.Paragraph;
import info.magnolia.cms.beans.config.Template;
import info.magnolia.cms.beans.runtime.ParagraphRenderer;
import info.magnolia.cms.beans.runtime.TemplateRenderer;
import info.magnolia.cms.core.Content;
import info.magnolia.cms.core.NodeData;
import info.magnolia.context.MgnlContext;
import info.magnolia.context.WebContext;

import java.io.IOException;
import java.io.Writer;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.sourceforge.stripes.controller.DispatcherServlet;
import net.sourceforge.stripes.controller.StripesConstants;
import net.sourceforge.stripes.controller.StripesRequestWrapper;
import net.sourceforge.stripes.exception.StripesServletException;

import org.apache.commons.lang.StringUtils;


/**
 * <p>
 * A Magnolia paragraph renderer that delegates to Stripes actions. Most of the code is just a cut and paste from
 * Stripes' {@link DispatcherServlet}, adapted to work withing magnolia by:
 * </p>
 * <ul>
 * <li>wrap the request in order to modify the request path and provide a custom requestDispatcher</li>
 * <li>wrap the response in order to provide a custom Writer</li>
 * <li>use a fake servlet instance/context to setup a Stripe context (there is no servlet here)</li>
 * <li>injiect any paragraph property as a parameter</li>
 * </ul>
 * <p>
 * <strong>Todo:</strong>
 * </p>
 * <ul>
 * <li>Handle multipart forms in request wrapper (should be easy to do)</li>
 * <li>A better way of handling multivalued properties in paragraph</li>
 * <li>Handle binary properties in paragraph</li>
 * </ul>
 * @author fgiust
 * @version $Id: StripesRenderer.java 910 2008-09-15 22:06:51Z fgiust $
 */
public class StripesRenderer implements ParagraphRenderer, TemplateRenderer
{

    private StripesDispatcherServlet stripesDispatcherServlet = new StripesDispatcherServlet();

    /**
     * {@inheritDoc}
     */
    public void renderTemplate(Template template, HttpServletRequest request, HttpServletResponse response)
        throws IOException, ServletException
    {
        final String templatePath = template.getPath();
        Map<String, String[]> nodeDataMap = contentToMap(MgnlContext.getAggregationState().getCurrentContent());

        renderCommon(templatePath, nodeDataMap, response.getWriter());
    }

    /**
     * {@inheritDoc}
     */
    public void render(Content content, Paragraph paragraph, Writer out) throws IOException
    {
        // @fgiust start custom magnolia setup

        final String templatePath = paragraph.getTemplatePath();
        Map<String, String[]> nodeDataMap = contentToMap(content);

        renderCommon(templatePath, nodeDataMap, out);
    }

    /**
     * {@inheritDoc}
     */
    protected void renderCommon(String templatePath, Map<String, String[]> nodeDataMap, Writer out) throws IOException
    {

        WebContext webContext = (WebContext) MgnlContext.getInstance();
        HttpServletResponse response = new MgnlStripesResponseWrapper(webContext.getResponse(), out);

        HttpServletRequest request;
        try
        {

            HttpServletRequest originalreq = new MgnlStripesRequestWrapper(((WebContext) MgnlContext.getInstance())
                .getRequest(), templatePath, nodeDataMap, out);

            request = new StripesRequestWrapper(originalreq);
        }
        catch (StripesServletException e)
        {
            throw new RuntimeException(e);
        }

        request.setAttribute(StripesConstants.REQ_ATTR_INCLUDE_PATH, templatePath);

        try
        {
            stripesDispatcherServlet.doPost(request, response);
        }
        catch (ServletException e)
        {
            throw new RuntimeException(e);
        }
    }

    /**
     * @param content paragraph node
     * @return a map of Strings (converted nodedata)
     */
    @SuppressWarnings("unchecked")
    protected Map<String, String[]> contentToMap(Content content)
    {
        Collection<NodeData> paragraphsData = content.getNodeDataCollection();
        Map<String, String[]> nodeDataMap = new HashMap<String, String[]>();
        for (NodeData nodeData : paragraphsData)
        {
            String name = StringUtils.replaceChars(nodeData.getName(), "{}", "[]");
            String value = nodeData.getString();
            if (StringUtils.contains(name, "multiple"))
            {
                nodeDataMap.put(name, StringUtils.split(value, "\n"));
            }
            else
            {
                nodeDataMap.put(name, new String[]{value });
            }
        }
        return nodeDataMap;
    }

    /**
     * just needed to make the protected doPost() accessible
     * @author fgiust
     * @version $Id: StripesRenderer.java 910 2008-09-15 22:06:51Z fgiust $
     */
    protected class StripesDispatcherServlet extends DispatcherServlet
    {

        /**
         * Stable serialVersionUID.
         */
        private static final long serialVersionUID = 42L;

        /**
         * {@inheritDoc}
         */
        @Override
        public void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException
        {
            super.doPost(request, response);
        }

        /**
         * {@inheritDoc}
         */
        @Override
        public ServletContext getServletContext()
        {
            return ((WebContext) MgnlContext.getInstance()).getServletContext();
        }

    }

}
