/**
 * Copyright Openmind http://www.openmindonline.it
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */
package it.openutils.magnoliastripes;

import info.magnolia.cms.beans.config.Paragraph;
import info.magnolia.cms.beans.config.ParagraphManager;

import java.util.HashSet;
import java.util.Set;

import net.sourceforge.stripes.action.ActionBean;
import net.sourceforge.stripes.controller.NameBasedActionResolver;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * ActionResolver that extends <code>NameBasedActionResolver</code>, registering any found Stripe action as a
 * Magnolia paragraph.
 * @author fgiust
 * @version $Id: $
 */
public class MgnlActionResolver extends NameBasedActionResolver
{

    /**
     * Logger.
     */
    private static Logger log = LoggerFactory.getLogger(MgnlActionResolver.class);

    /**
     * Configured Stripes paragraphs.
     */
    private static Set<Paragraph> paragraphs = new HashSet<Paragraph>();

    /**
     * {@inheritDoc}
     */
    @Override
    protected void addActionBean(Class< ? extends ActionBean> clazz)
    {
        String binding = getUrlBinding(clazz);

        // Only process the class if it's properly annotated
        if (binding != null)
        {
            String dialogName = actionNameToParagraphName(binding);
            collectStripesParagraphs(dialogName, binding);
            super.addActionBean(clazz);
        }
    }

    /**
     * Generate a paragraph name from a Stripes binding. This method will take the last token after "/", strip any
     * extension and convert everything to lowercase.
     * @param binding Stripe action binding
     * @return paragraph name
     */
    protected String actionNameToParagraphName(String binding)
    {
        String dialogName = StringUtils.lowerCase(StringUtils.substringBeforeLast(StringUtils.substringAfterLast(
            binding,
            "/"), "."));
        return dialogName;
    }

    /**
     * Registers a Magnolia paragraph which will delegate to a Stripe action.
     * @param name dialog name
     * @param binding Stripes action binding
     */
    @SuppressWarnings("unchecked")
    private void collectStripesParagraphs(String name, String binding)
    {
        Paragraph paragraph = new Paragraph();

        paragraph.setName(name);
        paragraph.setTitle("paragraph." + name + ".title");
        paragraph.setDescription("paragraph." + name + ".description");
        paragraph.setDialog(name);
        paragraph.setTemplatePath(binding);
        paragraph.setType("stripes");
        paragraphs.add(paragraph);

        log.info("Registering stripes paragraph {}", paragraph.getName()); //$NON-NLS-1$
        ParagraphManager.getInstance().getParagraphs().put(paragraph.getName(), paragraph);
    }

    @SuppressWarnings("unchecked")
    public static void registerParagraphs()
    {
        for (Paragraph paragraph : paragraphs)
        {
            log.info("Registering stripes paragraph {}", paragraph.getName()); //$NON-NLS-1$
            ParagraphManager.getInstance().getParagraphs().put(paragraph.getName(), paragraph);
        }
    }

}
