/**
 *
 * Criteria API for Magnolia CMS (http://www.openmindlab.com/lab/products/mgnlcriteria.html)
 * Copyright(C) 2009-2011, Openmind S.r.l. http://www.openmindonline.it
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.sourceforge.openutils.mgnlcriteria.jcr.query.xpath.utils;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * A utility class to escape xpath strings
 * @author fgiust
 * @author fgrilli
 * @version $Id: XPathTextUtils.java 3285 2011-01-24 18:10:19Z fgiust $
 */
public final class XPathTextUtils
{

    private static Logger log = LoggerFactory.getLogger(XPathTextUtils.class);

    private XPathTextUtils()
    {
    }

    /**
     * Convert a string to an XPath 2.0 string literal, suitable for inclusion in a query. See JSR-170 spec v1.0, Sec.
     * 6.6.4.9.
     * @param str Any string.
     * @return A valid XPath 2.0 string literal, including enclosing quotes.
     */
    public static String stringToXPathLiteral(String str)
    {
        if (StringUtils.isEmpty(str))
        {
            return str;
        }
        // Single quotes needed for jcr:contains()
        return str.replaceAll("'", "\"");
    }

    /**
     * Convert a string to a JCR search expression literal, suitable for use in jcr:contains() (inside XPath queries).
     * The characters - and " have special meaning, and may be escaped with a backslash to obtain their literal value.
     * See JSR-170 spec v1.0, Sec. 6.6.5.2.
     * @param str Any string.
     * @return A valid XPath 2.0 string literal suitable for use in jcr:contains(), including enclosing quotes.
     */
    public static String stringToJCRSearchExp(String str)
    {
        if (StringUtils.isEmpty(str))
        {
            return str;
        }

        String parseString = StringUtils.trimToEmpty(str);

        // workaround for https://issues.apache.org/jira/browse/JCR-2732
        parseString = StringUtils.replaceEach(parseString, new String[]{":)", ":(" }, new String[]{": )", ": (" });

        /*
         * http://lucene.apache.org/java/2_4_0/queryparsersyntax.html#Escaping%20Special%20Characters
         * http://www.javalobby.org/java/forums/t86124.html
         */
        String escapeChars = "[\\\\+\\-\\!\\(\\)\\:\\^\\]\\{\\}\\~\\*\\?\"\\[\\]|]";
        parseString = parseString.replaceAll(escapeChars, "\\\\$0");
        parseString = parseString.replaceAll("\'", "\'\'");

        // workaround for https://issues.apache.org/jira/browse/JCR-2733
        if (StringUtils.startsWith(parseString, "OR "))
        {
            parseString = parseString.replaceFirst("\\bOR\\b", "\"OR\"");
        }

        return parseString;

    }

    /**
     * @param path to encode eg //my//path/2009//*
     * @return String encoded path eg //my//path/_x0032_009//*
     */
    public static String encodeDigitsInPath(String path)
    {
        log.debug("path to encode is {}", path);
        if (StringUtils.isBlank(path))
        {
            String msg = "path cannot be a null or empty string";
            log.error(msg);
            throw new IllegalArgumentException(msg);
        }

        StringBuilder encodedPath = new StringBuilder(path.length());

        boolean inXpathCondition = false;

        // TODO maybe a more robust check is needed
        for (int i = 0; i < path.length(); ++i)
        {
            char ch = path.charAt(i);

            if (i > 0 && path.charAt(i - 1) == '/' && Character.isDigit(ch))
            {
                encodedPath.append("_x" + StringUtils.leftPad(Integer.toHexString(ch), 4, '0') + "_");
            }
            else if (!inXpathCondition && ch == ' ')
            {
                encodedPath.append("_x0020_");
            }
            else
            {

                if (ch == '[')
                {
                    inXpathCondition = true;
                }
                else if (inXpathCondition && ch == ']')
                {
                    inXpathCondition = false;
                }
                encodedPath.append(ch);
            }
        }
        log.debug("returning encoded path {}", encodedPath);
        return encodedPath.toString();
    }
}