/**
 *
 * Magnolia Criteria API (http://www.openmindlab.com/lab/products/mgnlcriteria.html)
 * Copyright(C) 2009-2010, Openmind S.r.l. http://www.openmindonline.it
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.sourceforge.openutils.mgnlcriteria.tests;

import info.magnolia.cms.beans.config.ContentRepository;
import info.magnolia.cms.beans.config.PropertiesInitializer;
import info.magnolia.cms.core.Path;
import info.magnolia.cms.core.SystemProperty;
import info.magnolia.cms.util.ClasspathResourcesUtil;
import info.magnolia.context.MgnlContext;
import info.magnolia.context.SystemContext;
import info.magnolia.context.SystemRepositoryStrategy;
import info.magnolia.importexport.BootstrapUtil;
import info.magnolia.test.mock.MockContext;
import info.magnolia.test.mock.MockUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.util.Map;

import javax.jcr.ImportUUIDBehavior;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.core.jndi.BindableRepositoryFactory;
import org.apache.log4j.Level;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testng.annotations.AfterClass;
import org.testng.annotations.BeforeClass;


/**
 * TestNG version of Magnolia RepositoryTestcase
 * @author fgiust
 * @version $Id: RepositoryTestNgTestcase.java 2912 2010-08-27 18:50:47Z fgiust $
 */

public abstract class RepositoryTestNgTestcase
{

    protected static final String REPO_CONF_PROPERTY = "magnolia.repositories.config";

    protected static final String JACKRABBIT_REPO_CONF_PROPERTY = "magnolia.repositories.jackrabbit.config";

    protected static final String EXTRACTED_REPO_CONF_FILE = "target/repositories.xml";

    protected static final String EXTRACTED_JACKRABBIT_REPO_CONF_FILE = "target/repo-conf/extracted.xml";

    private boolean autoStart = true;

    private String repositoryConfigFileName;

    private String jackrabbitRepositoryConfigFileName;

    private boolean quiet = true;

    protected boolean isAutoStart()
    {
        return this.autoStart;
    }

    protected void setAutoStart(boolean autoStart)
    {
        this.autoStart = autoStart;
    }

    protected void setRepositoryConfigFileName(String repositoryConfigFileName)
    {
        this.repositoryConfigFileName = repositoryConfigFileName;
    }

    protected void setJackrabbitRepositoryConfigFileName(String jackrabbitRepositoryConfigFileName)
    {
        this.jackrabbitRepositoryConfigFileName = jackrabbitRepositoryConfigFileName;
    }

    protected boolean isQuiet()
    {
        return this.quiet;
    }

    protected void setQuiet(boolean quiet)
    {
        this.quiet = quiet;
    }

    /**
     * Logger.
     */
    protected Logger log = LoggerFactory.getLogger(getClass());

    @BeforeClass
    protected void setUp() throws Exception
    {
        // ignore mapping warnings
        org.apache.log4j.Logger.getLogger(ContentRepository.class).setLevel(Level.ERROR);

        // ComponentsTestUtil.clear();
        initDefaultImplementations();
        setMagnoliaProperties();
        initContext();
        workaroundJCR1778();

        if (isAutoStart())
        {
            cleanUp();
            startRepository();
        }
    }

    protected void initContext()
    {
        MockUtil.initMockContext();
    }

    @AfterClass
    protected void tearDown() throws Exception
    {
        if (isAutoStart())
        {
            shutdownRepository(true);
        }
        SystemProperty.getProperties().clear();

        // ComponentsTestUtil.clear();
        SystemProperty.getProperties().clear();
        MgnlContext.setInstance(null);
    }

    protected void setMagnoliaProperties() throws Exception
    {
        setMagnoliaProperties(getMagnoliaPropertiesStream());
    }

    protected void setMagnoliaProperties(InputStream propertiesStream) throws IOException
    {
        try
        {
            SystemProperty.getProperties().load(propertiesStream);
        }
        finally
        {
            IOUtils.closeQuietly(propertiesStream);
        }
    }

    protected InputStream getMagnoliaPropertiesStream() throws IOException
    {
        return this.getClass().getResourceAsStream(getMagnoliaPropertiesFileName());
    }

    protected String getMagnoliaPropertiesFileName()
    {
        return "/test-magnolia.properties";
    }

    protected void initDefaultImplementations() throws IOException
    {
        PropertiesInitializer.getInstance().loadBeanProperties();
        PropertiesInitializer.getInstance().loadAllModuleProperties();
    }

    /**
     * @deprecated - workaround until JCR-1778 is fixed
     * @see <a href="https://issues.apache.org/jira/browse/JCR-1778">JCR-1778</a>
     */
    static void workaroundJCR1778() throws NoSuchFieldException, IllegalAccessException
    {
        final Field cacheField = BindableRepositoryFactory.class.getDeclaredField("cache");
        cacheField.setAccessible(true);
        final Map cache = (Map) cacheField.get(null);
        cache.clear();
    }

    protected void modifyContextesToUseRealRepository()
    {
        // create a mock web context with same repository acquiring strategy as the system context
        SystemContext systemContext = (SystemContext) MgnlContext.getSystemContext();
        SystemRepositoryStrategy repositoryStrategy = new SystemRepositoryStrategy(systemContext);

        // update the mock context
        ((MockContext) systemContext).setRepositoryStrategy(repositoryStrategy);
        ((MockContext) MgnlContext.getInstance()).setRepositoryStrategy(repositoryStrategy);
    }

    protected void startRepository() throws Exception
    {
        final org.apache.log4j.Logger logger = org.apache.log4j.Logger.getLogger("info.magnolia");
        final Level originalLogLevel = logger.getLevel();
        if (this.isQuiet())
        {
            logger.setLevel(Level.WARN);
        }
        ContentRepository.REPOSITORY_USER = SystemProperty.getProperty("magnolia.connection.jcr.userId");
        ContentRepository.REPOSITORY_PSWD = SystemProperty.getProperty("magnolia.connection.jcr.password");

        extractConfigFile(REPO_CONF_PROPERTY, getRepositoryConfigFileStream(), EXTRACTED_REPO_CONF_FILE);
        extractConfigFile(
            JACKRABBIT_REPO_CONF_PROPERTY,
            getJackrabbitRepositoryConfigFileStream(),
            EXTRACTED_JACKRABBIT_REPO_CONF_FILE);

        ContentRepository.init();

        modifyContextesToUseRealRepository();

        logger.setLevel(originalLogLevel);
    }

    protected void extractConfigFile(String propertyName, InputStream configFileStream, String extractToPath)
        throws Exception
    {
        String targetFilename = Path.getAbsoluteFileSystemPath(extractToPath);
        File targetFile = new File(targetFilename);
        // extract resource to the filesystem (jackrabbit can't use a stream)
        new File(targetFile.getParent()).mkdirs();
        IOUtils.copy(configFileStream, new FileOutputStream(targetFile));
        SystemProperty.setProperty(propertyName, extractToPath);
    }

    protected InputStream getRepositoryConfigFileStream() throws Exception
    {
        String configFile = getRepositoryConfigFileName();
        return ClasspathResourcesUtil.getResource(configFile).openStream();
    }

    protected InputStream getJackrabbitRepositoryConfigFileStream() throws Exception
    {
        String configFile = getJackrabbitRepositoryConfigFileName();
        return ClasspathResourcesUtil.getResource(configFile).openStream();
    }

    protected String getRepositoryConfigFileName()
    {
        if (StringUtils.isEmpty(repositoryConfigFileName))
        {
            repositoryConfigFileName = SystemProperty.getProperty(REPO_CONF_PROPERTY);
        }
        return repositoryConfigFileName;
    }

    protected String getJackrabbitRepositoryConfigFileName()
    {
        if (StringUtils.isEmpty(jackrabbitRepositoryConfigFileName))
        {
            jackrabbitRepositoryConfigFileName = SystemProperty.getProperty(JACKRABBIT_REPO_CONF_PROPERTY);
        }
        return jackrabbitRepositoryConfigFileName;
    }

    protected void shutdownRepository(boolean cleanup) throws IOException
    {
        final org.apache.log4j.Logger logger = org.apache.log4j.Logger.getLogger("info.magnolia");
        final Level originalLogLevel = logger.getLevel();
        if (this.isQuiet())
        {
            logger.setLevel(Level.WARN);
        }
        MgnlContext.release();
        MgnlContext.getSystemContext().release();
        ContentRepository.shutdown();
        if (cleanup)
        {
            cleanUp();
        }
        logger.setLevel(originalLogLevel);
    }

    protected void cleanUp() throws IOException
    {
        FileUtils.deleteDirectory(new File(SystemProperty.getProperty("magnolia.repositories.home")));
    }

    protected void bootstrapSingleResource(String resource) throws Exception
    {
        BootstrapUtil.bootstrap(new String[]{resource }, ImportUUIDBehavior.IMPORT_UUID_COLLISION_THROW);
    }

    protected void bootstrap(ClasspathResourcesUtil.Filter filter) throws Exception
    {
        String[] resourcesToBootstrap = ClasspathResourcesUtil.findResources(filter);
        BootstrapUtil.bootstrap(resourcesToBootstrap, ImportUUIDBehavior.IMPORT_UUID_COLLISION_THROW);
    }

}
