/**
 *
 * Magnolia Criteria API (http://www.openmindlab.com/lab/products/mgnlcriteria.html)
 * Copyright(C) 2009-2010, Openmind S.r.l. http://www.openmindonline.it
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.sourceforge.openutils.mgnlcriteria.jcr.query;

import info.magnolia.cms.beans.config.ContentRepository;
import info.magnolia.cms.core.Content;
import info.magnolia.cms.core.HierarchyManager;
import info.magnolia.cms.core.ItemType;
import info.magnolia.cms.util.NodeDataUtil;
import info.magnolia.context.MgnlContext;

import java.util.Calendar;

import net.sourceforge.openutils.mgnlcriteria.jcr.query.criterion.Order;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.criterion.Restrictions;
import net.sourceforge.openutils.mgnlcriteria.tests.RepositoryTestNgTestcase;

import org.apache.commons.lang.StringUtils;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;


/**
 * @author dschivo
 * @version $Id: CriteriaTest.java 2900 2010-08-25 13:15:14Z diego_schivo $
 */
public class CriteriaTest extends RepositoryTestNgTestcase
{

    /**
     * {@inheritDoc}
     */
    @Override
    @BeforeClass
    protected void setUp() throws Exception
    {
        setRepositoryConfigFileName("/crit-repository/test-repositories.xml");
        setJackrabbitRepositoryConfigFileName("/crit-repository/jackrabbit-test-configuration.xml");

        super.setUp();

        // Nodes in this workspace:
        // - pets (title=Pets)
        // --- cats (title=Cats)
        // ----- 1 (title=Leo, petType=cat, birthDate=2000-09-07)
        // ----- 7 (title=Samantha, petType=cat, birthDate=1995-09-04)
        // ----- 8 (title=Max, petType=cat, birthDate=1995-09-04)
        // ----- 13 (title=Sly, petType=cat, birthDate=2002-06-08)
        // --- dogs (title=Dogs)
        // ----- 3 (title=Rosy, petType=dog, birthDate=2001-04-17)
        // ----- 4 (title=Jewel, petType=dog, birthDate=2000-03-07)
        // ----- 10 (title=Mulligan, petType=dog, birthDate=1997-02-24)
        // ----- 12 (title=Lucky, petType=dog, birthDate=2000-06-24)
        // --- lizards (title=Lizards)
        // ----- 5 (title=Iggy, petType=lizard, birthDate=2000-11-30)
        // --- snakes (title=Snakes)
        // ----- 6 (title=George, petType=snake, birthDate=2000-01-20)
        // --- birds (title=Birds)
        // ----- 9 (title=Lucky, petType=bird, birthDate=1999-08-06)
        // ----- 11 (title=Freddy, petType=bird, birthDate=2000-03-09)
        // --- hamsters (title=Hamsters)
        // ----- 2 (title=Basil, petType=hamster, birthDate=2002-08-06)
        bootstrapSingleResource("/crit-bootstrap/website.pets.xml");
        MgnlContext.getHierarchyManager(ContentRepository.WEBSITE).save();
    }

    /**
     * Tests the xpath query statement produced by a Criteria instance.
     * @throws Exception
     */
    @Test
    public void testToXpathExpression() throws Exception
    {
        Criteria criteria = toXpathExpressionJavadocExampleCriteria();

        String expectedStmt = "//pets//*"
            + "[((jcr:contains(@title, 'Lucky')) and (@petType='dog')"
            + " and (@birthDate >=xs:dateTime('1999-01-01T00:00:00.000+00:00')"
            + " and @birthDate <=xs:dateTime('2001-12-31T23:59:59.999+00:00')))]"
            + " order by @title descending";
        String actualStmt = criteria.toXpathExpression();
        Assert.assertEquals(StringUtils.remove(actualStmt, ' '), StringUtils.remove(expectedStmt, ' '));
    }

    /**
     * @throws Exception
     */
    @Test
    public void testExecuteTrivial() throws Exception
    {
        HierarchyManager hm = MgnlContext.getHierarchyManager(ContentRepository.WEBSITE);
        Content node = hm.getContent("/pets");
        Assert.assertEquals(node.getTitle(), "Pets");

        Criteria criteria = JCRCriteriaFactory.createCriteria().setWorkspace(ContentRepository.WEBSITE).setBasePath(
            "/jcr:root/*").add(Restrictions.eq("@jcr:primaryType", ItemType.CONTENT.getSystemName())).add(
            Restrictions.eq("@title", "Pets"));
        AdvancedResult result = criteria.execute();
        ResultIterator<AdvancedResultItem> iterator = result.getItems();
        Assert.assertTrue(iterator.hasNext());
        Content resultNode = iterator.next();
        Assert.assertEquals(resultNode.getTitle(), "Pets");
    }

    /**
     * Tests the query execution of a Criteria instance.
     * @throws Exception
     */
    @Test
    public void testExecute() throws Exception
    {
        Criteria criteria = toXpathExpressionJavadocExampleCriteria();

        AdvancedResult result = criteria.execute();
        Assert.assertEquals(result.getTotalSize(), 1);

        ResultIterator<AdvancedResultItem> iterator = result.getItems();
        Assert.assertEquals(iterator.getSize(), 1);
        Assert.assertEquals(iterator.next().getName(), "12");
    }

    /**
     * @return
     */
    private Criteria toXpathExpressionJavadocExampleCriteria()
    {
        Calendar begin = Calendar.getInstance();
        begin.set(1999, Calendar.JANUARY, 1);
        Calendar end = Calendar.getInstance();
        end.set(2001, Calendar.DECEMBER, 31);

        Criteria criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.contains("@title", "Lucky"))
            .add(Restrictions.eq("@petType", "dog"))
            .add(Restrictions.betweenDates("@birthDate", begin, end))
            .addOrder(Order.desc("@title"));
        return criteria;
    }

    /**
     * Tests pagination of results.
     * @throws Exception
     */
    @Test
    public void testSetFirstResultAndMaxResults() throws Exception
    {
        Calendar begin = Calendar.getInstance();
        begin.set(1999, Calendar.JANUARY, 1);
        Calendar end = Calendar.getInstance();
        end.set(2001, Calendar.DECEMBER, 31);

        Criteria criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.betweenDates("@birthDate", begin, end))
            .addOrder(Order.asc("@birthDate"))
            .setFirstResult(5)
            .setMaxResults(5);

        AdvancedResult result = criteria.execute();
        // first page:
        // --- 9 (title=Lucky, petType=bird, birthDate=1999-08-06)
        // --- 6 (title=George, petType=snake, birthDate=2000-01-20)
        // --- 4 (title=Jewel, petType=dog, birthDate=2000-03-07)
        // --- 11 (title=Freddy, petType=bird, birthDate=2000-03-09)
        // --- 12 (title=Lucky, petType=dog, birthDate=2000-06-24)
        // second page:
        // --- 1 (title=Leo, petType=cat, birthDate=2000-09-07)
        // --- 5 (title=Iggy, petType=lizard, birthDate=2000-11-30)
        // --- 3 (title=Rosy, petType=dog, birthDate=2001-04-17)
        Assert.assertEquals(result.getTotalSize(), 8);

        ResultIterator<AdvancedResultItem> iterator = result.getItems();
        Assert.assertEquals(iterator.getSize(), 3);
        Assert.assertEquals(iterator.next().getName(), "1");
        Assert.assertEquals(iterator.next().getName(), "5");
        Assert.assertEquals(iterator.next().getName(), "3");
    }

    /**
     * Tests pagination of results.
     * @throws Exception
     */
    @Test
    public void testSetPaging() throws Exception
    {
        Calendar begin = Calendar.getInstance();
        begin.set(1999, Calendar.JANUARY, 1);
        Calendar end = Calendar.getInstance();
        end.set(2001, Calendar.DECEMBER, 31);

        Criteria criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.betweenDates("@birthDate", begin, end))
            .addOrder(Order.asc("@birthDate"))
            .setPaging(5, 2);

        AdvancedResult result = criteria.execute();
        // first page:
        // --- 9 (title=Lucky, petType=bird, birthDate=1999-08-06)
        // --- 6 (title=George, petType=snake, birthDate=2000-01-20)
        // --- 4 (title=Jewel, petType=dog, birthDate=2000-03-07)
        // --- 11 (title=Freddy, petType=bird, birthDate=2000-03-09)
        // --- 12 (title=Lucky, petType=dog, birthDate=2000-06-24)
        // second page:
        // --- 1 (title=Leo, petType=cat, birthDate=2000-09-07)
        // --- 5 (title=Iggy, petType=lizard, birthDate=2000-11-30)
        // --- 3 (title=Rosy, petType=dog, birthDate=2001-04-17)
        Assert.assertEquals(result.getTotalSize(), 8);

        ResultIterator<AdvancedResultItem> iterator = result.getItems();
        Assert.assertEquals(iterator.getSize(), 3);
        Assert.assertEquals(iterator.next().getName(), "1");
        Assert.assertEquals(iterator.next().getName(), "5");
        Assert.assertEquals(iterator.next().getName(), "3");
    }

    /**
     * Tests ordering, both ascending and descending.
     * @throws Exception
     */
    @Test
    public void testAddOrder() throws Exception
    {
        Criteria criteria;
        ResultIterator<AdvancedResultItem> iterator;
        Calendar birthDate;

        // gets the oldest pet (ascending order)
        criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.isNotNull("@petType"))
            .addOrder(Order.asc("@birthDate"));
        iterator = criteria.execute().getItems();
        // ----- 7 (title=Samantha, petType=cat, birthDate=1995-09-04)
        // ----- 8 (title=Max, petType=cat, birthDate=1995-09-04)
        Assert.assertTrue(iterator.hasNext());
        birthDate = NodeDataUtil.getDate(iterator.next(), "birthDate", null);
        Assert.assertEquals(birthDate.get(Calendar.YEAR), 1995);
        Assert.assertEquals(birthDate.get(Calendar.MONTH) + 1, 9);
        Assert.assertEquals(birthDate.get(Calendar.DAY_OF_MONTH), 4);

        // gets the youngest pet (descending order)
        criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.isNotNull("@petType"))
            .addOrder(Order.desc("@birthDate"));
        iterator = criteria.execute().getItems();
        // ----- 2 (title=Basil, petType=hamster, birthDate=2002-08-06)
        Assert.assertTrue(iterator.hasNext());
        birthDate = NodeDataUtil.getDate(iterator.next(), "birthDate", null);
        Assert.assertEquals(birthDate.get(Calendar.YEAR), 2002);
        Assert.assertEquals(birthDate.get(Calendar.MONTH) + 1, 8);
        Assert.assertEquals(birthDate.get(Calendar.DAY_OF_MONTH), 6);
    }

    /**
     * Tests multiple ordering, playing on the fact that the two oldests pets are born on the same date but have
     * different name.
     * @throws Exception
     */
    @Test
    public void testAddOrderMultiple() throws Exception
    {
        Criteria criteria;
        ResultIterator<AdvancedResultItem> iterator;

        // order by @birthDate ascending, @title ascending
        criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.isNotNull("@petType"))
            .addOrder(Order.asc("@birthDate"))
            .addOrder(Order.asc("@title"));
        iterator = criteria.execute().getItems();
        // ----- 8 (title=Max, petType=cat, birthDate=1995-09-04)
        // ----- 7 (title=Samantha, petType=cat, birthDate=1995-09-04)
        Assert.assertTrue(iterator.hasNext());
        Assert.assertEquals(iterator.next().getTitle(), "Max");
        Assert.assertTrue(iterator.hasNext());
        Assert.assertEquals(iterator.next().getTitle(), "Samantha");

        // order by @birthDate ascending, @title descending
        criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.isNotNull("@petType"))
            .addOrder(Order.asc("@birthDate"))
            .addOrder(Order.desc("@title"));
        iterator = criteria.execute().getItems();
        // ----- 7 (title=Samantha, petType=cat, birthDate=1995-09-04)
        // ----- 8 (title=Max, petType=cat, birthDate=1995-09-04)
        Assert.assertTrue(iterator.hasNext());
        Assert.assertEquals(iterator.next().getTitle(), "Samantha");
        Assert.assertTrue(iterator.hasNext());
        Assert.assertEquals(iterator.next().getTitle(), "Max");
    }

    @Test
    public void testDateComparison() throws Exception
    {
        Criteria criteria;
        ResultIterator<AdvancedResultItem> iterator;
        Content node;
        Calendar date;

        criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.isNotNull("@petType"))
            .add(Restrictions.eq("@title", "Leo"));
        iterator = criteria.execute().getItems();
        Assert.assertTrue(iterator.hasNext());
        node = iterator.next();
        Assert.assertEquals(node.getTitle(), "Leo");

        date = (Calendar) node.getMetaData().getCreationDate().clone();
        date.add(Calendar.DAY_OF_YEAR, 1);
        criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.isNotNull("@petType"))
            .add(Restrictions.eq("@title", "Leo"));
        criteria.add(Restrictions.lt("MetaData/@mgnl:creationdate", date));
        iterator = criteria.execute().getItems();
        Assert.assertTrue(iterator.hasNext());
        node = iterator.next();
        Assert.assertEquals(node.getTitle(), "Leo");

        date = (Calendar) node.getMetaData().getCreationDate().clone();
        date.add(Calendar.HOUR, 1);
        criteria = JCRCriteriaFactory
            .createCriteria()
            .setWorkspace(ContentRepository.WEBSITE)
            .setBasePath("/pets")
            .add(Restrictions.isNotNull("@petType"))
            .add(Restrictions.eq("@title", "Leo"));
        criteria.add(Restrictions.lt("MetaData/@mgnl:creationdate", date));
        iterator = criteria.execute().getItems();
        Assert.assertTrue(iterator.hasNext());
        node = iterator.next();
        Assert.assertEquals(node.getTitle(), "Leo");
    }
}
