/**
 *
 * Magnolia Criteria API (http://www.openmindlab.com/lab/products/mgnlcriteria.html)
 * Copyright(C) 2009-2010, Openmind S.r.l. http://www.openmindonline.it
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.sourceforge.openutils.mgnlcriteria.advanced;

import info.magnolia.cms.beans.config.ContentRepository;
import info.magnolia.context.MgnlContext;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.AdvancedResult;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.AdvancedResultItem;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.Criteria;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.JCRCriteriaFactory;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.ResultIterator;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.criterion.Order;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.criterion.Restrictions;
import net.sourceforge.openutils.mgnlcriteria.tests.RepositoryTestNgTestcase;

import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;


/**
 * Tests pagination in criteria queries.
 * @author dschivo
 */
public class PaginationTest extends RepositoryTestNgTestcase
{

    /**
     * {@inheritDoc}
     */
    @Override
    @BeforeClass
    protected void setUp() throws Exception
    {
        setRepositoryConfigFileName("/crit-repository/test-repositories.xml");
        setJackrabbitRepositoryConfigFileName("/crit-repository/jackrabbit-test-configuration.xml");

        super.setUp();

        // Titles of the nodes in this workspace:
        // - Letters
        // --- A
        // --- B
        // --- C
        // --- ...
        // --- X
        // --- Y
        // --- Z
        bootstrapSingleResource("/crit-bootstrap/website.letters.xml");
        MgnlContext.getHierarchyManager(ContentRepository.WEBSITE).save();
    }

    /**
     * Retrieves all letters.
     * @throws Exception
     */
    @Test
    public void testNoPagination() throws Exception
    {
        Criteria criteria = JCRCriteriaFactory.createCriteria().setWorkspace(ContentRepository.WEBSITE);
        criteria.setBasePath("/letters");
        criteria.add(Restrictions.eq("@jcr:primaryType", "mgnl:content"));
        criteria.addOrder(Order.asc("@title"));
        AdvancedResult advResult = criteria.execute();

        Assert.assertEquals(advResult.getTotalSize(), 26);

        ResultIterator<AdvancedResultItem> resultIterator = advResult.getItems();
        Assert.assertEquals(resultIterator.getSize(), 26);

        Assert.assertEquals(resultIterator.next().getTitle(), "A");

        AdvancedResultItem content = null;
        while (resultIterator.hasNext())
        {
            content = resultIterator.next();
        }
        Assert.assertEquals(content.getTitle(), "Z");
    }

    /**
     * Retrieves letters from the tenth on.
     * @throws Exception
     */
    @Test
    public void testSetFirstResult() throws Exception
    {
        Criteria criteria = JCRCriteriaFactory.createCriteria().setWorkspace(ContentRepository.WEBSITE);
        criteria.setBasePath("/letters");
        criteria.add(Restrictions.eq("@jcr:primaryType", "mgnl:content"));
        criteria.addOrder(Order.asc("@title"));
        criteria.setFirstResult(9);
        AdvancedResult advResult = criteria.execute();

        Assert.assertEquals(advResult.getTotalSize(), 26);

        ResultIterator<AdvancedResultItem> resultIterator = advResult.getItems();
        Assert.assertEquals(resultIterator.getSize(), 17);

        Assert.assertEquals(resultIterator.next().getTitle(), "J");
    }

    /**
     * Retrieves the first ten letters.
     * @throws Exception
     */
    @Test
    public void testSetMaxResults() throws Exception
    {
        Criteria criteria = JCRCriteriaFactory.createCriteria().setWorkspace(ContentRepository.WEBSITE);
        criteria.setBasePath("/letters");
        criteria.add(Restrictions.eq("@jcr:primaryType", "mgnl:content"));
        criteria.addOrder(Order.asc("@title"));
        criteria.setMaxResults(10);
        AdvancedResult advResult = criteria.execute();

        Assert.assertEquals(advResult.getTotalSize(), 26);

        ResultIterator<AdvancedResultItem> resultIterator = advResult.getItems();
        Assert.assertEquals(resultIterator.getSize(), 10);

        AdvancedResultItem content = null;
        while (resultIterator.hasNext())
        {
            content = resultIterator.next();
        }
        Assert.assertEquals(content.getTitle(), "J");
    }

    /**
     * Retrieves the third page of 5 letters.
     * @throws Exception
     */
    @Test
    public void testSetPaging() throws Exception
    {
        Criteria criteria = JCRCriteriaFactory.createCriteria().setWorkspace(ContentRepository.WEBSITE);
        criteria.setBasePath("/letters");
        criteria.add(Restrictions.eq("@jcr:primaryType", "mgnl:content"));
        criteria.addOrder(Order.asc("@title"));
        criteria.setPaging(5, 3);
        AdvancedResult advResult = criteria.execute();

        Assert.assertEquals(advResult.getTotalSize(), 26);

        ResultIterator<AdvancedResultItem> resultIterator = advResult.getItems();

        Assert.assertEquals(resultIterator.getSize(), 5);
        Assert.assertEquals(resultIterator.next().getTitle(), "K");
        Assert.assertEquals(resultIterator.next().getTitle(), "L");
        Assert.assertEquals(resultIterator.next().getTitle(), "M");
        Assert.assertEquals(resultIterator.next().getTitle(), "N");
        Assert.assertEquals(resultIterator.next().getTitle(), "O");
    }

    /**
     * Retrieves the fourth page of 8 letters.
     * @throws Exception
     */
    @Test
    public void testSetPagingLastPage() throws Exception
    {
        Criteria criteria = JCRCriteriaFactory.createCriteria().setWorkspace(ContentRepository.WEBSITE);
        criteria.setBasePath("/letters");
        criteria.add(Restrictions.eq("@jcr:primaryType", "mgnl:content"));
        criteria.addOrder(Order.asc("@title"));
        criteria.setPaging(8, 4);
        AdvancedResult advResult = criteria.execute();

        Assert.assertEquals(advResult.getTotalSize(), 26);

        ResultIterator<AdvancedResultItem> resultIterator = advResult.getItems();

        Assert.assertEquals(resultIterator.getSize(), 2);
        Assert.assertEquals(resultIterator.next().getTitle(), "Y");
        Assert.assertEquals(resultIterator.next().getTitle(), "Z");
    }

    /**
     * Retrieves the last page of 3 letters.
     * @throws Exception
     */
    @Test
    public void testSetPagingEvenPage() throws Exception
    {
        Criteria criteria = JCRCriteriaFactory.createCriteria().setWorkspace(ContentRepository.WEBSITE);
        criteria.setBasePath("/letters");
        criteria.add(Restrictions.eq("@jcr:primaryType", "mgnl:content"));
        criteria.addOrder(Order.asc("@title"));
        criteria.setPaging(5, 1);

        AdvancedResult advResult = criteria.execute();
        Assert.assertEquals(advResult.getTotalSize(), 26);
        Assert.assertEquals(advResult.getNumberOfPages(), 6);

        ResultIterator<AdvancedResultItem> resultIterator = advResult.getItems();
        Assert.assertEquals(resultIterator.getSize(), 5);
    }
}
