/**
 *
 * Magnolia Criteria API (http://www.openmindlab.com/lab/products/mgnlcriteria.html)
 * Copyright(C) 2009-2010, Openmind S.r.l. http://www.openmindonline.it
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.sourceforge.openutils.mgnlcriteria.advanced;

import info.magnolia.cms.beans.config.ContentRepository;
import info.magnolia.context.MgnlContext;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import net.sourceforge.openutils.mgnlcriteria.jcr.query.AdvancedResult;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.AdvancedResultItem;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.Criteria;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.JCRCriteriaFactory;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.JCRQueryException;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.ResultIterator;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.criterion.Order;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.criterion.Restrictions;
import net.sourceforge.openutils.mgnlcriteria.tests.RepositoryTestNgTestcase;

import org.apache.commons.lang.StringUtils;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;


/**
 * @author dschivo
 * @version $Id: JcrContainsCriteriaSearchTest.java 2914 2010-08-28 08:14:49Z fgiust $
 */
public class JcrContainsCriteriaSearchTest extends RepositoryTestNgTestcase
{

    /**
     * {@inheritDoc}
     */
    @Override
    @BeforeClass
    protected void setUp() throws Exception
    {
        setRepositoryConfigFileName("/crit-repository/test-repositories.xml");
        setJackrabbitRepositoryConfigFileName("/crit-repository/jackrabbit-test-configuration.xml");

        super.setUp();

        // Titles of the nodes in this workspace:
        // - hello test? world
        // - hello te?st world
        // - hello "Milano" world
        // - lorem
        // - lorem ipsum
        // - dolor sit
        // - dolor sit amet
        bootstrapSingleResource("/crit-bootstrap/website.contains.xml");
        MgnlContext.getHierarchyManager(ContentRepository.WEBSITE).save();
    }

    @Test
    public void testLoremAndIpsum() throws Exception
    {
        Criteria criteria = criteria("lorem ipsum", false);
        AdvancedResult advResult = criteria.execute();
        Assert.assertNotNull(advResult);
        Assert.assertEquals(advResult.getTotalSize(), 1);
        ResultIterator<AdvancedResultItem> items = advResult.getItems();
        AdvancedResultItem item = items.next();
        Assert.assertEquals(item.getTitle(), "lorem ipsum");
    }

    @Test
    public void testLoremAndNotIpsum() throws Exception
    {
        Criteria criteria = criteria("lorem -ipsum", false);
        AdvancedResult advResult = criteria.execute();
        Assert.assertNotNull(advResult);
        Assert.assertEquals(advResult.getTotalSize(), 1);
        ResultIterator<AdvancedResultItem> items = advResult.getItems();
        AdvancedResultItem item = items.next();
        Assert.assertEquals(item.getTitle(), "lorem");
    }

    @Test
    public void testIpsumOrAmet() throws Exception
    {
        Criteria criteria = criteria("ipsum OR amet", false);
        AdvancedResult advResult = criteria.execute();
        Assert.assertNotNull(advResult);
        Assert.assertEquals(advResult.getTotalSize(), 2);
        ResultIterator<AdvancedResultItem> items = advResult.getItems();
        List<String> titles = new ArrayList<String>();
        while (items.hasNext())
        {
            titles.add(items.next().getTitle());
        }
        Collections.sort(titles);
        Assert.assertEquals(titles.get(0), "dolor sit amet");
        Assert.assertEquals(titles.get(1), "lorem ipsum");
    }

    @Test
    public void testIpsumOrSitAndAmet() throws Exception
    {
        Criteria criteria = criteria("ipsum OR \"sit AND amet\"", false);
        AdvancedResult advResult = criteria.execute();
        Assert.assertNotNull(advResult);
        Assert.assertEquals(advResult.getTotalSize(), 2);
        ResultIterator<AdvancedResultItem> items = advResult.getItems();
        List<String> titles = new ArrayList<String>();
        while (items.hasNext())
        {
            titles.add(items.next().getTitle());
        }
        Collections.sort(titles);
        Assert.assertEquals(titles.get(0), "dolor sit amet");
        Assert.assertEquals(titles.get(1), "lorem ipsum");
    }

    @Test
    public void testTest1() throws Exception
    {
        String textEnteredByUser = "test?";
        Criteria criteria = criteria(textEnteredByUser, true);
        Assert.assertEquals(
            StringUtils.remove(criteria.toXpathExpression(), ' '),
            "//*[((@jcr:primaryType='mgnl:content')and(jcr:contains(@title,'test\\?')))]orderby@jcr:scoredescending");
        AdvancedResult advResult = null;
        try
        {
            advResult = criteria.execute();
        }
        catch (JCRQueryException e)
        {
            Assert.fail("Invalid query. " + e.getMessage());
        }
        Assert.assertNotNull(advResult);
        Assert.assertEquals(advResult.getTotalSize(), 1);
        ResultIterator<AdvancedResultItem> items = advResult.getItems();
        AdvancedResultItem item = items.next();
        Assert.assertEquals(item.getTitle(), "hello test? world");
    }

    @Test
    public void testTest2() throws Exception
    {
        String textEnteredByUser = "te?st";
        Criteria criteria = criteria(textEnteredByUser, true);
        Assert.assertEquals(
            StringUtils.remove(criteria.toXpathExpression(), ' '),
            "//*[((@jcr:primaryType='mgnl:content')and(jcr:contains(@title,'te\\?st')))]orderby@jcr:scoredescending");
        AdvancedResult advResult = null;
        try
        {
            advResult = criteria.execute();
        }
        catch (JCRQueryException e)
        {
            Assert.fail("Invalid query. " + e.getMessage());
        }
        Assert.assertNotNull(advResult);
        Assert.assertEquals(advResult.getTotalSize(), 1);
        ResultIterator<AdvancedResultItem> items = advResult.getItems();
        AdvancedResultItem item = items.next();
        Assert.assertEquals(item.getTitle(), "hello te?st world");
    }

    @Test
    public void testMilano() throws Exception
    {
        String textEnteredByUser = "\"Milano\"";
        Criteria criteria = criteria(textEnteredByUser, true);
        Assert.assertEquals(
            StringUtils.remove(criteria.toXpathExpression(), ' '),
                "//*[((@jcr:primaryType='mgnl:content')and(jcr:contains(@title,'\\\"Milano\\\"')))]orderby@jcr:scoredescending");
        AdvancedResult advResult = null;
        try
        {
            advResult = criteria.execute();
        }
        catch (JCRQueryException e)
        {
            Assert.fail("Invalid query. " + e.getMessage());
        }
        Assert.assertNotNull(advResult);
        Assert.assertEquals(advResult.getTotalSize(), 1);
        ResultIterator<AdvancedResultItem> items = advResult.getItems();
        AdvancedResultItem item = items.next();
        Assert.assertEquals(item.getTitle(), "hello \"Milano\" world");
    }

    private Criteria criteria(String titleSearch, boolean escape)
    {
        Criteria criteria = JCRCriteriaFactory.createCriteria().setWorkspace(ContentRepository.WEBSITE);
        criteria.setBasePath(StringUtils.EMPTY);
        criteria.add(Restrictions.eq("@jcr:primaryType", "mgnl:content"));
        criteria.add(Restrictions.contains("@title", titleSearch, escape));
        criteria.addOrder(Order.desc("@jcr:score"));
        return criteria;
    }
}
