/**
 *
 * Magnolia Criteria API (http://www.openmindlab.com/lab/products/mgnlcriteria.html)
 * Copyright(C) 2009-2010, Openmind S.r.l. http://www.openmindonline.it
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.sourceforge.openutils.mgnlcriteria.advanced;

import info.magnolia.cms.beans.config.ContentRepository;
import info.magnolia.context.MgnlContext;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import net.sourceforge.openutils.mgnlcriteria.jcr.query.AdvancedResult;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.AdvancedResultItem;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.Criteria;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.JCRCriteriaFactory;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.ResultIterator;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.criterion.Criterion;
import net.sourceforge.openutils.mgnlcriteria.jcr.query.criterion.Restrictions;
import net.sourceforge.openutils.mgnlcriteria.tests.RepositoryTestNgTestcase;

import org.apache.commons.lang.StringUtils;
import org.testng.Assert;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.Test;


/**
 * Tests criteria queries with different values of basePath parameter.
 * @author dschivo
 * @version $Id: BasePathTest.java 2923 2010-08-29 08:58:29Z fgiust $
 */
public class BasePathTest extends RepositoryTestNgTestcase
{

    /**
     * {@inheritDoc}
     */
    @Override
    @BeforeClass
    protected void setUp() throws Exception
    {
        setRepositoryConfigFileName("/crit-repository/test-repositories.xml");
        setJackrabbitRepositoryConfigFileName("/crit-repository/jackrabbit-test-configuration.xml");

        super.setUp();

        // Titles of the nodes in this workspace:
        // - Criteria
        // --- AbstractCriteriaImpl
        // ----- AbstractMagnoliaCriteriaImpl
        // ------- MagnoliaCriteriaImpl
        // --------- MagnoliaCriteriaWithLimitImpl
        // ----- AdvancedCriteriaImpl
        // --- AdvancedCriteriaImpl
        // --- TranslatableCriteria
        // ----- AbstractCriteriaImpl
        // ------- AbstractMagnoliaCriteriaImpl
        // --------- MagnoliaCriteriaImpl
        // ----------- MagnoliaCriteriaWithLimitImpl
        // ------- AdvancedCriteriaImpl
        bootstrapSingleResource("/crit-bootstrap/website.Criteria.xml");
        MgnlContext.getHierarchyManager(ContentRepository.WEBSITE).save();
    }

    /**
     * Passing a null basePath should search the entire repository.
     * @throws Exception
     */
    @Test
    public void testNullBasePath() throws Exception
    {
        Collection<String> paths = searchPaths(null, "AdvancedCriteriaImpl");
        Assert.assertNotNull(paths);
        Assert.assertEquals(paths.size(), 3);
        Assert.assertTrue(paths.contains("/Criteria/AbstractCriteriaImpl/AdvancedCriteriaImpl"));
        Assert.assertTrue(paths.contains("/Criteria/AdvancedCriteriaImpl"));
        Assert.assertTrue(paths.contains("/Criteria/TranslatableCriteria/AbstractCriteriaImpl/AdvancedCriteriaImpl"));
    }

    /**
     * Passing an empty basePath should search the entire repository.
     * @throws Exception
     */
    @Test
    public void testEmptyBasePath() throws Exception
    {
        Collection<String> paths = searchPaths(StringUtils.EMPTY, "AdvancedCriteriaImpl");
        Assert.assertNotNull(paths);
        Assert.assertEquals(paths.size(), 3);
        Assert.assertTrue(paths.contains("/Criteria/AbstractCriteriaImpl/AdvancedCriteriaImpl"));
        Assert.assertTrue(paths.contains("/Criteria/AdvancedCriteriaImpl"));
        Assert.assertTrue(paths.contains("/Criteria/TranslatableCriteria/AbstractCriteriaImpl/AdvancedCriteriaImpl"));
    }

    /**
     * Passing an xpath query ending with /* as the basePath should search the children.
     * @throws Exception
     */
    @Test
    public void testSearchXpathBasePathWithSingleSlash() throws Exception
    {
        Collection<String> paths = searchPaths("//Criteria/AbstractCriteriaImpl/*", StringUtils.EMPTY);
        Assert.assertNotNull(paths);
        Assert.assertEquals(paths.size(), 2);
        Assert.assertTrue(paths.contains("/Criteria/AbstractCriteriaImpl/AbstractMagnoliaCriteriaImpl"));
        Assert.assertTrue(paths.contains("/Criteria/AbstractCriteriaImpl/AdvancedCriteriaImpl"));
    }

    /**
     * Passing an xpath query ending with //* as the basePath should search the descendants.
     * @throws Exception
     */
    @Test
    public void testSearchXpathBasePathWithDoubleSlash() throws Exception
    {
        Collection<String> paths = searchPaths("//Criteria/TranslatableCriteria//*", "AdvancedCriteriaImpl");
        Assert.assertNotNull(paths);
        Assert.assertEquals(paths.size(), 1);
        Assert.assertTrue(paths.contains("/Criteria/TranslatableCriteria/AbstractCriteriaImpl/AdvancedCriteriaImpl"));
    }

    /**
     * Passing an handle as the basePath should search the descendants.
     * @throws Exception
     */
    @Test
    public void testSearchHandleBasePath() throws Exception
    {
        Collection<String> paths = searchPaths("/Criteria/TranslatableCriteria", "AdvancedCriteriaImpl");
        Assert.assertNotNull(paths);
        Assert.assertEquals(paths.size(), 1);
        Assert.assertTrue(paths.contains("/Criteria/TranslatableCriteria/AbstractCriteriaImpl/AdvancedCriteriaImpl"));
    }

    /**
     * Passing an handle ending with / as the basePath should search the descendants. This test makes sure that the
     * resulting xpath query does not end with ///*
     * @throws Exception
     */
    @Test
    public void testSearchHandleBasePathWithTrailingSlash() throws Exception
    {
        Collection<String> paths = searchPaths("/Criteria/TranslatableCriteria/", "AdvancedCriteriaImpl");
        Assert.assertNotNull(paths);
        Assert.assertEquals(paths.size(), 1);
        Assert.assertTrue(paths.contains("/Criteria/TranslatableCriteria/AbstractCriteriaImpl/AdvancedCriteriaImpl"));
    }

    private Collection<String> searchPaths(String basePath, String title)
    {
        Criteria criteria = JCRCriteriaFactory.createCriteria().setWorkspace(ContentRepository.WEBSITE);
        criteria.setBasePath(basePath);
        criteria.add(Restrictions.eq(Criterion.JCR_PRIMARYTYPE, "mgnl:content"));
        if (!StringUtils.isEmpty(title))
        {
            criteria.add(Restrictions.eq("@title", title));
        }
        AdvancedResult advResult = criteria.execute();
        ResultIterator<AdvancedResultItem> items = advResult.getItems();
        List<String> paths = new ArrayList<String>();
        while (items.hasNext())
        {
            paths.add(items.next().getHandle());
        }
        return paths;
    }
}
