/**
 *
 * Magnolia Criteria API (http://www.openmindlab.com/lab/products/mgnlcriteria.html)
 * Copyright(C) 2009-2010, Openmind S.r.l. http://www.openmindonline.it
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package net.sourceforge.openutils.mgnlcriteria.jcr.query.lucene;

import info.magnolia.cms.security.AccessManager;
import info.magnolia.cms.security.Permission;
import info.magnolia.cms.util.SimpleUrlPattern;
import info.magnolia.context.MgnlContext;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;

import javax.jcr.RepositoryException;

import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.core.SessionImpl;
import org.apache.jackrabbit.core.query.lucene.MatchAllDocsQuery;
import org.apache.jackrabbit.core.query.lucene.QueryDecoratorSupport;
import org.apache.jackrabbit.core.query.lucene.SearchIndex;
import org.apache.lucene.search.BooleanClause.Occur;
import org.apache.lucene.search.Query;


/**
 * Decorates a lucene query by adding constraints for acl rules.
 * @author dschivo
 * @version $Id: AclQueryDecorator.java 2922 2010-08-29 07:20:05Z fgiust $
 */
public class AclQueryDecorator extends QueryDecoratorSupport
{

    private final SessionImpl session;

    /**
     *
     */
    public AclQueryDecorator(SessionImpl session, SearchIndex index)
    {
        super(index);
        this.session = session;
    }

    public Query applyAcl(Query query) throws RepositoryException
    {
        // creates a lucene query for each acl rule
        List<Object[]> aclQueries = new ArrayList<Object[]>();
        AccessManager accessManager = MgnlContext.getAccessManager(session.getWorkspace().getName());
        List<Permission> permissions = accessManager.getPermissionList();
        if (!permissions.isEmpty())
        {
            try
            {
                Field patternStringField = SimpleUrlPattern.class.getDeclaredField("patternString");
                patternStringField.setAccessible(true);
                for (Permission permission : permissions)
                {
                    if (permission.getPattern() instanceof SimpleUrlPattern)
                    {
                        String pattern = (String) patternStringField.get(permission.getPattern());
                        String[] tokens = StringUtils.splitPreserveAllTokens(pattern, '/');
                        if (tokens.length > 2 && "".equals(tokens[0]) && "*".equals(tokens[tokens.length - 1]))
                        {
                            Query q = null;
                            for (int i = 1; i < tokens.length - 1; i++)
                            {
                                if (q == null)
                                {
                                    q = descendantSelfAxisQuery(jackrabbitTermQuery("_:PARENT"), nameQuery(tokens[i]));
                                }
                                else
                                {
                                    q = childAxisQuery(q, tokens[i]);
                                }
                            }
                            q = descendantSelfAxisQuery(booleanQuery(q), new MatchAllDocsQuery());
                            Occur o = (permission.getPermissions() & Permission.READ) != 0
                                ? Occur.MUST
                                : Occur.MUST_NOT;
                            aclQueries.add(new Object[]{q, o });
                        }
                    }
                }
            }
            catch (SecurityException e)
            {
                throw new RepositoryException(e);
            }
            catch (IllegalArgumentException e)
            {
                throw new RepositoryException(e);
            }
            catch (NoSuchFieldException e)
            {
                throw new RepositoryException(e);
            }
            catch (IllegalAccessException e)
            {
                throw new RepositoryException(e);
            }
        }

        // combines the original lucene query with the acl ones
        if (!aclQueries.isEmpty())
        {
            Query[] qs = new Query[1 + aclQueries.size()];
            Occur[] os = new Occur[qs.length];
            qs[0] = query;
            os[0] = Occur.MUST;
            for (int i = 1; i < qs.length; i++)
            {
                Object[] qo = aclQueries.get(i - 1);
                qs[i] = (Query) qo[0];
                os[i] = (Occur) qo[1];
            }
            query = booleanQuery(qs, os);
        }

        return query;
    }
}
