/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.hibernate.security.services.impl;

import it.openutils.hibernate.security.dao.SecurityRuleDAO;
import it.openutils.hibernate.security.dataobject.SecurityRule;
import it.openutils.hibernate.security.filter.JavaBeanFilter;
import it.openutils.hibernate.security.services.SecurityRuleManager;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.hibernate.Filter;


/**
 * @author fcarone
 * @version $Id: SecurityRuleManagerImpl.java 729 2008-03-06 09:26:46Z fcarone $
 */
public class SecurityRuleManagerImpl implements SecurityRuleManager
{

    private SecurityRuleDAO securityRuleDAO;

    /**
     * Sets the securityRuleDAO.
     * @param securityRuleDAO the securityRuleDAO to set
     */
    public void setSecurityRuleDAO(SecurityRuleDAO securityRuleDAO)
    {
        this.securityRuleDAO = securityRuleDAO;
    }

    /**
     * {@inheritDoc}
     */
    public List<SecurityRule> findAll()
    {
        return securityRuleDAO.findAll();
    }

    /**
     * {@inheritDoc}
     */
    public List<SecurityRule> findFiltered(SecurityRule filter)
    {
        return securityRuleDAO.findFiltered(filter);
    }

    /**
     * {@inheritDoc}
     */
    public Filter getEntityFilterFromRules(String entity, List<SecurityRule> rules) throws SecurityException,
        ClassNotFoundException, InstantiationException, IllegalAccessException, NoSuchFieldException
    {
        return new JavaBeanFilter(entity, rules);
    }

    /**
     * {@inheritDoc}
     */
    public SecurityRule load(Long id)
    {
        return securityRuleDAO.load(id);
    }

    /**
     * {@inheritDoc}
     */
    public SecurityRule loadIfAvailable(Long id)
    {
        return securityRuleDAO.loadIfAvailable(id);
    }

    /**
     * {@inheritDoc}
     */
    public void delete(SecurityRule securityRule)
    {
        securityRule.setDeleted(true);
        securityRuleDAO.update(securityRule);
    }

    /**
     * {@inheritDoc}
     */
    public Long save(SecurityRule securityRule)
    {
        return securityRuleDAO.save(securityRule);
    }

    /**
     * {@inheritDoc}
     */
    public void saveOrUpdate(SecurityRule securityRule)
    {
        securityRuleDAO.saveOrUpdate(securityRule);
    }

    /**
     * {@inheritDoc}
     */
    public void update(SecurityRule securityRule)
    {
        securityRuleDAO.update(securityRule);
    }

    /**
     * {@inheritDoc}
     */
    public List<SecurityRule> getRulesForRoles(String entity, List<String> roles)
    {
        List<SecurityRule> rules = new ArrayList<SecurityRule>();
        for (String role : roles)
        {
            SecurityRule filter = new SecurityRule();
            filter.setEntity(entity);
            filter.setRole(role);
            rules.addAll(findFiltered(filter));
        }
        return rules;
    }

    /**
     * {@inheritDoc}
     */
    public List<SecurityRule> getRulesForRoles(Object entity, List<String> roles)
    {
        return findAllRules(entity, roles);
    }

    @SuppressWarnings("unchecked")
    private List<SecurityRule> findAllRules(Object object, List<String> roles)
    {
        List<SecurityRule> result = new ArrayList<SecurityRule>();

        Class objectClass = object.getClass();
        while (!objectClass.isAssignableFrom(Object.class))
        {
            result.addAll(getRulesForRoles(StringUtils.substringBefore(objectClass.getName(), "$$"), roles));

            objectClass = objectClass.getSuperclass();
        }
        return result;
    }
}
