/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.hibernate.security.filter;

import it.openutils.hibernate.security.dataobject.SecurityRule;
import it.openutils.hibernate.security.filter.utils.SecurityRuleUtils;
import it.openutils.hibernate.security.services.SecurityRuleManager;

import java.util.List;

import org.acegisecurity.Authentication;
import org.acegisecurity.ConfigAttributeDefinition;
import org.acegisecurity.vote.AccessDecisionVoter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ReflectiveMethodInvocation;


/**
 * @author fcarone
 * @version $Id: SecurityRuleAccessDecisionVoter.java 729 2008-03-06 09:26:46Z fcarone $
 *
 * This voter looks for rules based on the given object and votes ACCESS_ABSTAIN if no rules are found, ACCESS_DENIED
 * if rules do not match, ACCESS_GRANTED otherwise.
 *
 */
public class SecurityRuleAccessDecisionVoter extends SecurityRuleBaseHandler implements AccessDecisionVoter
{

    /**
     * Logger.
     */
    private static Logger log = LoggerFactory.getLogger(SecurityRuleAccessDecisionVoter.class);

    private SecurityRuleManager securityRuleManager;

    private SecurityRuleUtils ruleUtils = new SecurityRuleUtils();

    private boolean enabled = true;

    /**
     * {@inheritDoc}
     */
    public int vote(Authentication authentication, Object object, ConfigAttributeDefinition config)
    {
        if (!enabled)
        {
            log.debug("SecurityRuleAccessDecisionVoter disabled, abstaining.");
            return ACCESS_ABSTAIN;
        }
        List<String> roles = ruleUtils.getRolesFromAuthentication(authentication);

        if (object instanceof ReflectiveMethodInvocation)
        {
            ReflectiveMethodInvocation methodInvocation = ((ReflectiveMethodInvocation) object);
            for (Object argument : methodInvocation.getArguments())
            {
                log.debug("Evaluating argument {}", argument);
                List<SecurityRule> rules = securityRuleManager.getRulesForRoles(ruleUtils.getClassName(argument), roles);
                if (rules == null || rules.isEmpty())
                {
                    continue;
                }
                if (ruleUtils.checkPermissions(rules, config) && ruleUtils.checkRules(rules, argument))
                {
                    log.debug("Granting access.");
                    return ACCESS_GRANTED;
                }
                log.debug("Denying access.");
                return ACCESS_DENIED;
            }
        }
        log.debug("No rules found, abstaining.");
        return ACCESS_ABSTAIN;
    }

    /**
     * Sets the securityRuleManager.
     * @param securityRuleManager the securityRuleManager to set
     */
    public void setSecurityRuleManager(SecurityRuleManager securityRuleManager)
    {
        this.securityRuleManager = securityRuleManager;
    }


    /**
     * Sets the enabled.
     * @param enabled the enabled to set
     */
    public void setEnabled(boolean enabled)
    {
        this.enabled = enabled;
    }
}
