/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.hibernate.security.filter;

import it.openutils.hibernate.security.dataobject.PermissionEnum;
import it.openutils.hibernate.security.dataobject.SecurityRule;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Table;

import org.acegisecurity.util.FieldUtils;
import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Filter;
import org.hibernate.HibernateException;
import org.hibernate.engine.FilterDefinition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author fcarone
 * @version $Id: JavaBeanFilter.java 729 2008-03-06 09:26:46Z fcarone $
 */
public class JavaBeanFilter implements Filter
{

    private FilterDefinition filterDefinition;

    /**
     * Logger.
     */
    private Logger log = LoggerFactory.getLogger(JavaBeanFilter.class);


    /**
     * @param bean The bean to set rules for
     * @param securityRules The list of {@link SecurityRule}s to apply.
     * @throws ClassNotFoundException If the bean class has not been found
     * @throws InstantiationException If the bean doesn't contain the no-arg constructor
     * @throws IllegalAccessException If the bean properties cannot be accessed
     * @throws SecurityException If the bean class cannot be accessed
     * @throws NoSuchFieldException If the property contained in the security rule refers to a bean non-existent field
     */
    @SuppressWarnings("unchecked")
    public JavaBeanFilter(String bean, List<SecurityRule> securityRules)
        throws ClassNotFoundException,
        InstantiationException,
        IllegalAccessException,
        SecurityException,
        NoSuchFieldException
    {
        Class< ? extends Object> beanClass = Class.forName(bean, true, this.getClass().getClassLoader());

        if (!beanClass.isAnnotationPresent(Entity.class))
        {
            throw new IllegalArgumentException("Class " + bean + " must contain the @Entity annotation.");
        }
        if (!(beanClass.isAnnotationPresent(Table.class) || beanClass
            .isAnnotationPresent(org.hibernate.annotations.Table.class)))
        {
            throw new IllegalArgumentException("Class " + bean + " must contain the @Table annotation.");
        }

        Map<String, String> propertyColumnMap = new HashMap<String, String>();

        StringBuffer filterDefCondition = new StringBuffer();
        String filterName = StringUtils.EMPTY;

        Map<String, List<SecurityRule>> roleRuleMap = new LinkedHashMap<String, List<SecurityRule>>();
        for (SecurityRule securityRule : securityRules)
        {
            if (!securityRule.isEnabled())
            {
                continue;
            }

            if (!roleRuleMap.containsKey(securityRule.getRole()))
            {
                roleRuleMap.put(securityRule.getRole(), new ArrayList<SecurityRule>());
            }
            roleRuleMap.get(securityRule.getRole()).add(securityRule);
        }

        for (Map.Entry<String, List<SecurityRule>> entry : roleRuleMap.entrySet())
        {
            filterName += entry.getKey();

            List<SecurityRule> rules = entry.getValue();
            if (rules == null || rules.isEmpty())
            {
                log.debug("No rules defined for role {}", entry.getKey());
                continue;
            }

            if (!rulesContainLoad(rules))
            {
                log.debug("No LOAD rules defined for role {}", entry.getKey());
                continue;
            }

            if (!StringUtils.isEmpty(filterDefCondition.toString()))
            {
                filterDefCondition.append(" OR ");
            }
            filterDefCondition.append("(");
            StringBuffer subFilterCond = new StringBuffer();
            for (SecurityRule securityRule : rules)
            {
                if (!securityRule.getPermissions().contains(PermissionEnum.LOAD))
                {
                    log.debug("Skipping rule {} since it is not related to LOAD.", securityRule);
                    continue;
                }
                String property = securityRule.getProperty();
                filterName += property;

                Field field = FieldUtils.getField(beanClass, property);

                // @todo: annotations may also be defined on getters/setters...
                propertyColumnMap.put(property, field.getAnnotation(Column.class).name());
                if (!StringUtils.isEmpty(subFilterCond.toString()))
                {
                    subFilterCond.append(" AND ");
                }
                String modifier = null;
                String startQuote = null;
                String endQuote = null;

                if (String.class.isAssignableFrom(field.getType()))
                {
                    startQuote = "\'";
                    endQuote = startQuote;
                }
                else if (Number.class.isAssignableFrom(field.getType()))
                {
                    startQuote = StringUtils.EMPTY;
                    endQuote = StringUtils.EMPTY;
                }

                switch (securityRule.getModifier())
                {
                    case EQUALS :
                        modifier = " = ";
                        break;

                    case NOT :
                        modifier = " != ";
                        break;

                    default :
                        throw new IllegalArgumentException("Modifier " + securityRule.getModifier() + "not recognized.");
                }
                subFilterCond.append(field.getAnnotation(Column.class).name());
                subFilterCond.append(modifier);
                if (StringUtils.isNotEmpty(startQuote))
                {
                    subFilterCond.append(startQuote);
                }
                subFilterCond.append(StringEscapeUtils.escapeSql(securityRule.getValue()));
                if (StringUtils.isNotEmpty(endQuote))
                {
                    subFilterCond.append(endQuote);
                }

                filterName += securityRule.getValue();
            }
            filterDefCondition.append(subFilterCond);
            filterDefCondition.append(")");
        }

        // filtername is unique, but untraceable
        this.filterDefinition = new FilterDefinition(Integer.toString(filterName.hashCode()), filterDefCondition
            .toString(), new HashMap());
    }

    /**
     * @param rules
     * @return
     */
    private boolean rulesContainLoad(List<SecurityRule> rules)
    {
        for (SecurityRule rule : rules)
        {
            if (rule.getPermissions().contains(PermissionEnum.LOAD))
            {
                return true;
            }
        }
        return false;
    }

    /**
     * {@inheritDoc}
     */
    public FilterDefinition getFilterDefinition()
    {
        return this.filterDefinition;
    }

    /**
     * {@inheritDoc}
     */
    public String getName()
    {
        return this.filterDefinition.getFilterName();
    }

    /**
     * {@inheritDoc}
     */
    public Filter setParameter(String name, Object value)
    {
        return this;
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public Filter setParameterList(String name, Collection values)
    {
        return this;
    }

    /**
     * {@inheritDoc}
     */
    public Filter setParameterList(String name, Object[] values)
    {
        return this;
    }

    /**
     * {@inheritDoc}
     */
    public void validate() throws HibernateException
    {
        //
    }

}
