/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.hibernate.security.filter;

import it.openutils.hibernate.security.dataobject.SecurityRule;
import it.openutils.hibernate.security.filter.utils.SecurityRuleUtils;
import it.openutils.hibernate.security.services.SecurityRuleManager;

import java.util.ArrayList;
import java.util.List;

import org.acegisecurity.Authentication;
import org.acegisecurity.ConfigAttributeDefinition;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author fcarone
 * @version $Id: SecurityRulePredicate.java 705 2008-02-26 10:56:55Z fcarone $
 */
public class SecurityRulePredicate implements Predicate
{

    /**
     * Logger.
     */
    private Logger log = LoggerFactory.getLogger(SecurityRulePredicate.class);

    private Authentication authentication;

    private SecurityRuleManager securityRuleManager;

    private ConfigAttributeDefinition configAttribute;

    private SecurityRuleUtils ruleUtils = new SecurityRuleUtils();

    private boolean denyIfNoRulesFound = true;

    /**
     * @param authentication The current authentication
     * @param configAttribute The config attribute of the current method invocation interceptor
     * @param securityRuleManager The security rule manager to retrieve rules from
     */
    public SecurityRulePredicate(
        Authentication authentication,
        ConfigAttributeDefinition configAttribute,
        SecurityRuleManager securityRuleManager)
    {
        this.authentication = authentication;
        this.securityRuleManager = securityRuleManager;
        this.configAttribute = configAttribute;
    }

    /**
     * {@inheritDoc} If we return false here, the element will be removed from the original collection.
     */
    @SuppressWarnings("unchecked")
    public boolean evaluate(Object object)
    {
        List<String> roles = ruleUtils.getRolesFromAuthentication(authentication);
        List<SecurityRule> rules = findAllRules(object, roles);

        if (rules == null || rules.isEmpty())
        {
            if (denyIfNoRulesFound)
            {
                log.debug("Returning false since no rules found and denyIfNoRulesFound is true.");
                return false;
            }
            log.debug("Returning true since no rules found and denyIfNoRulesFound is false.");
            return true;

        }
        log.debug("Evaluating rules.");
        return (ruleUtils.checkPermissions(rules, configAttribute) && ruleUtils.checkRules(rules, object));
    }

    @SuppressWarnings("unchecked")
    private List<SecurityRule> findAllRules(Object object, List<String> roles)
    {
        List<SecurityRule> result = new ArrayList<SecurityRule>();

        Class objectClass = object.getClass();
        while (!objectClass.isAssignableFrom(Object.class))
        {
            result.addAll(securityRuleManager.getRulesForRoles(
                StringUtils.substringBefore(objectClass.getName(), "$$"),
                roles));

            objectClass = objectClass.getSuperclass();
        }
        return result;
    }

    /**
     * Sets the denyIfNoRulesFound.
     * @param denyIfNoRulesFound the denyIfNoRulesFound to set
     */
    public void setDenyIfNoRulesFound(boolean denyIfNoRulesFound)
    {
        this.denyIfNoRulesFound = denyIfNoRulesFound;
    }

}
