/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.hibernate.security.filter;

import it.openutils.hibernate.security.services.SecurityRuleManager;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.acegisecurity.AccessDeniedException;
import org.acegisecurity.Authentication;
import org.acegisecurity.ConfigAttributeDefinition;
import org.acegisecurity.afterinvocation.AfterInvocationProvider;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author fcarone
 * @version $Id: SecurityRuleAfterInvocationHandler.java 705 2008-02-26 10:56:55Z fcarone $
 */
public class SecurityRuleAfterInvocationHandler extends SecurityRuleBaseHandler implements AfterInvocationProvider
{

    /**
     * Logger.
     */
    private Logger log = LoggerFactory.getLogger(SecurityRuleAfterInvocationHandler.class);

    private SecurityRuleManager securityRuleManager;

    private boolean denyIfNoRulesFound = true;

    private boolean enabled = true;

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public Object decide(Authentication authentication, Object object, ConfigAttributeDefinition config,
        Object returnedObject) throws AccessDeniedException
    {
        if (!enabled)
        {
            log.debug("SecurityRuleAfterInvocationHandler disabled, returning.");
            return returnedObject;
        }
        if (returnedObject == null)
        {
            return null;
        }
        SecurityRulePredicate predicate = new SecurityRulePredicate(authentication, config, securityRuleManager);
        predicate.setDenyIfNoRulesFound(denyIfNoRulesFound);

        if (returnedObject instanceof Collection)
        {
            CollectionUtils.filter((Collection) returnedObject, predicate);
        }
        else if (returnedObject.getClass().isArray())
        {
            Object[] objectArray = (Object[]) returnedObject;
            List<Object> objectArrayList = Arrays.asList(objectArray);
            CollectionUtils.filter(objectArrayList, predicate);
        }
        else if (!predicate.evaluate(returnedObject))
        {
            throw new AccessDeniedException("Access denied");
        }
        return returnedObject;
    }

    /**
     * Sets the securityRuleManager.
     * @param securityRuleManager the securityRuleManager to set
     */
    public void setSecurityRuleManager(SecurityRuleManager securityRuleManager)
    {
        this.securityRuleManager = securityRuleManager;
    }

    /**
     * Sets the denyIfNoRulesFound.
     * @param denyIfNoRulesFound the denyIfNoRulesFound to set
     */
    public void setDenyIfNoRulesFound(boolean denyIfNoRulesFound)
    {
        this.denyIfNoRulesFound = denyIfNoRulesFound;
    }

    /**
     * Sets the enabled.
     * @param enabled the enabled to set
     */
    public void setEnabled(boolean enabled)
    {
        this.enabled = enabled;
    }

}
