/*
 * Copyright (c) Openmind.  All rights reserved. http://www.openmindonline.it
 */
package it.openutils.hibernate.security.filter.utils;

import it.openutils.hibernate.security.dataobject.PermissionEnum;
import it.openutils.hibernate.security.dataobject.SecurityRule;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.acegisecurity.Authentication;
import org.acegisecurity.ConfigAttribute;
import org.acegisecurity.ConfigAttributeDefinition;
import org.acegisecurity.GrantedAuthority;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * @author fcarone
 * @version $Id: SecurityRuleUtils.java 672 2008-02-20 17:14:54Z fcarone $
 */
public class SecurityRuleUtils
{
    /**
     * Logger.
     */
    private Logger log = LoggerFactory.getLogger(SecurityRuleUtils.class);

    /**
     * @param rules The list of rules to check
     * @param config The config attribute to check the rules against
     * @return True if any of the rules matches the given config attribute, false otherwise.
     */
    @SuppressWarnings("unchecked")
    public boolean checkPermissions(List<SecurityRule> rules, ConfigAttributeDefinition config)
    {
        log.debug("Evaluation permissions");
        Iterator iterator = config.getConfigAttributes();
        while (iterator.hasNext())
        {
            String attribute = ((ConfigAttribute) iterator.next()).getAttribute();
            for (SecurityRule rule : rules)
            {
                for (PermissionEnum permission : rule.getPermissions())
                {
                    if (StringUtils.equals(permission.getValue(), attribute))
                    {
                        log.debug("Matching permission: {}", permission.getValue());
                        return true;
                    }
                }
            }
        }
        log.debug("No matching permissions found.");
        return false;
    }

    /**
     * @param rules The list of rules to check
     * @param argument The object to check the rules against
     * @return True if any of the rules matches the given object, false otherwise.
     */
    public boolean checkRules(List<SecurityRule> rules, Object argument)
    {
        log.debug("Evaluating rules.");
        try
        {
            for (SecurityRule rule : rules)
            {
                String objProperty = BeanUtils.getSimpleProperty(argument, rule.getProperty());
                switch (rule.getModifier())
                {
                    case EQUALS:
                        if (StringUtils.equals(objProperty, rule.getValue()))
                        {
                            log.debug("Matching rule found: {}", rule);
                            return true;
                        }
                        break;
                    case NOT:
                        if (!StringUtils.equals(objProperty, rule.getValue()))
                        {
                            log.debug("Matching rule found: {}", rule);
                            return true;
                        }
                        break;                        
                    default:
                        throw new RuntimeException("Modifier " + rule.getModifier() + " is not recognized");
                }
            }
        }
        catch (NoSuchMethodException e)
        {
            log.error("{}", e);
        }
        catch (IllegalAccessException e)
        {
            log.error("{}", e);
        }
        catch (InvocationTargetException e)
        {
            log.error("{}", e);
        }
        log.debug("No matching rules found.");
        return false;
    }


    /**
     * @param authentication The authentication method
     * @return The collection of roles contained in the authentication
     */
    public List<String> getRolesFromAuthentication(Authentication authentication)
    {
        List<String> roles = new ArrayList<String>();
        for (GrantedAuthority authority : authentication.getAuthorities())
        {
            log.debug("Granted authority for user {}: {}", authentication.getName(), authority.getAuthority());
            roles.add(authority.getAuthority());
        }
        return roles;
    }


    /**
     * @param object The object to get the name from
     * @return The object name
     */
    public String getClassName(Object object)
    {
        // is this class a CGLib proxy?
        if (StringUtils.contains(object.getClass().getName(), "$$"))
        {
            return StringUtils.substringBefore(object.getClass().getName(), "$$");
        }
        return object.getClass().getName();
    }
}
