/*
 * Copyright (c) Openmind.  All rights reserved. http://www.openmindonline.it
 */
package it.openutils.hibernate.security.filter;

import it.openutils.hibernate.security.dataobject.SecurityRule;
import it.openutils.hibernate.security.filter.utils.SecurityRuleUtils;
import it.openutils.hibernate.security.services.SecurityRuleManager;

import java.util.ArrayList;
import java.util.List;

import org.acegisecurity.Authentication;
import org.acegisecurity.ConfigAttributeDefinition;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang.StringUtils;


/**
 * @author fcarone
 * @version $Id: SecurityRulePredicate.java 680 2008-02-21 14:57:26Z fcarone $
 */
public class SecurityRulePredicate implements Predicate
{

    private Authentication authentication;

    private SecurityRuleManager securityRuleManager;

    private ConfigAttributeDefinition configAttribute;

    private SecurityRuleUtils ruleUtils = new SecurityRuleUtils();

    /**
     * @param authentication The current authentication
     * @param configAttribute The config attribute of the current method invocation interceptor
     * @param securityRuleManager The security rule manager to retrieve rules from
     */
    public SecurityRulePredicate(
        Authentication authentication,
        ConfigAttributeDefinition configAttribute,
        SecurityRuleManager securityRuleManager)
    {
        this.authentication = authentication;
        this.securityRuleManager = securityRuleManager;
        this.configAttribute = configAttribute;
    }

    /**
     * {@inheritDoc} If we return false here, the element will be removed from the original collection.
     */
    @SuppressWarnings("unchecked")
    public boolean evaluate(Object object)
    {
        List<String> roles = ruleUtils.getRolesFromAuthentication(authentication);
        List<SecurityRule> rules = findAllRules(object, roles);

        // @todo: this should be configurable
        // denyAll by default
        if (rules == null || rules.isEmpty())
        {
            return false;
        }
        return (ruleUtils.checkPermissions(rules, configAttribute) && ruleUtils.checkRules(rules, object));
    }

    @SuppressWarnings("unchecked")
    // @todo: needs unit test methods
    private List<SecurityRule> findAllRules(Object object, List<String> roles)
    {
        List<SecurityRule> result = new ArrayList<SecurityRule>();

        Class objectClass = object.getClass();
        while (!objectClass.isAssignableFrom(Object.class))
        {
            result.addAll(securityRuleManager.getRulesForRoles(
                StringUtils.substringBefore(objectClass.getName(), "$$"),
                roles));

            objectClass = objectClass.getSuperclass();
        }
        return result;
    }

}
