/*
 * Copyright Openmind http://www.openmindonline.it
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package it.openutils.hibernate.security.filter;

import it.openutils.hibernate.security.dataobject.SecurityRule;
import it.openutils.hibernate.security.filter.utils.SecurityRuleUtils;
import it.openutils.hibernate.security.services.SecurityRuleManager;

import java.util.List;

import org.acegisecurity.Authentication;
import org.acegisecurity.ConfigAttributeDefinition;
import org.acegisecurity.vote.AccessDecisionVoter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ReflectiveMethodInvocation;


/**
 * @author fcarone
 * @version $Id: SecurityRuleAccessDecisionVoter.java 672 2008-02-20 17:14:54Z fcarone $
 */
public class SecurityRuleAccessDecisionVoter extends SecurityRuleBaseHandler implements AccessDecisionVoter
{

    /**
     * Logger.
     */
    private static Logger log = LoggerFactory.getLogger(SecurityRuleAccessDecisionVoter.class);

    private SecurityRuleManager securityRuleManager;

    private SecurityRuleUtils ruleUtils = new SecurityRuleUtils();

    /**
     * {@inheritDoc}
     */
    public int vote(Authentication authentication, Object object, ConfigAttributeDefinition config)
    {
        List<String> roles = ruleUtils.getRolesFromAuthentication(authentication);

        if (object instanceof ReflectiveMethodInvocation)
        {
            ReflectiveMethodInvocation methodInvocation = ((ReflectiveMethodInvocation) object);
            for (Object argument : methodInvocation.getArguments())
            {
                log.debug("Evaluating argument {}", argument);
                List<SecurityRule> rules = securityRuleManager.getRulesForRoles(ruleUtils.getClassName(argument), roles);
                boolean permissionMatches = ruleUtils.checkPermissions(rules, config);
                if (!permissionMatches)
                {
                    return ACCESS_DENIED;
                }

                boolean areRulesMatching = ruleUtils.checkRules(rules, argument);
                if (!areRulesMatching)
                {
                    return ACCESS_DENIED;
                }
            }
        }

        return ACCESS_GRANTED;
    }

    /**
     * Sets the securityRuleManager.
     * @param securityRuleManager the securityRuleManager to set
     */
    public void setSecurityRuleManager(SecurityRuleManager securityRuleManager)
    {
        this.securityRuleManager = securityRuleManager;
    }
}
