/*****************************************************************************
 * net.openai.fsm.BooleanCondition
 *****************************************************************************
 * @author  JC on E
 * @date    9/18/2000
 * 2001 OpenAI Labs
 *****************************************************************************/

package net.openai.util.fsm;


/**
 * BooleanCondition class
 */
public final class BooleanCondition extends AbstractCondition {

    /** Our target boolean value. */
    private boolean targetValue = true;

    /**
     * Constructs a BooleanCondition with it's target value set to
     * <code>true</code>.
     */
    public BooleanCondition() {
	this(null, true);
    }

    /**
     * Constructs a BooleanCondition with the given target value.
     *
     * @param target The boolean value that will satisfy this condition.
     */
    public BooleanCondition(boolean target) {
	this(null, target);
    }

    /**
     * Constructs a BooleanCondition with the given target value.
     *
     * @param target The Boolean value that will satisfy this condition.
     */
    public BooleanCondition(Boolean target) {
	this(null, target);
    }

    /**
     * Constructs a BooleanCondtion with it's target value set to 
     * <code>true</code> and it's target state set to the given state.
     *
     * @param targetState The target state for this BooleanCondition.
     */
    public BooleanCondition(State targetState) {
	this(targetState, true);
    }

    /**
     * Constructs a BooleanCondition with the given target value and the
     * given target state.
     *
     * @param targetState The target state for this BooleanCondtion.
     * @param target      The boolean value that will satisfy this condition.
     */
    public BooleanCondition(State targetState, boolean target) {
	super(targetState);
	setTarget(target);
    }

    /**
     * Constructs a BooleanCondition with the given target value.
     *
     * @param targetState The target state for this BooleanCondition.
     * @param target      The Boolean value that will satisfy this condition.
     */
    public BooleanCondition(State targetState, Boolean target) {
	super(targetState);
	setTarget(target);
    }

    /**
     * Sets the target boolean value that will satisfy this condition.
     *
     * @param target The boolean value that will satisfy this condition.
     */
    public final void setTarget(boolean target) {
	targetValue = target;
    }

    /**
     * Sets the target Boolean value that will satisfy this condition.
     *
     * @param target The Boolean value that will satisfy this condition.
     */
    public final void setTarget(Boolean target) {
	targetValue = target.booleanValue();
    }

    /**
     * Returns the target value that will satisfy this condition.
     *
     * @return The target value that will satisfy this condition.
     */
    public final boolean getTarget() {
	return targetValue;
    }

    /**
     * Called to check if the conditional meets the criteria defined by this
     * state.  This is the only method to implement in order to used this
     * interface.
     *
     * @param conditional The object to check.
     * @return Implementors should return <code>true</code> if this condition
     *         is met by the Object <code>conditional</code> and
     *         <code>false</code> otherwise.
     */
    public final boolean satisfiedBy(Object conditional) {
	if(conditional == null)
	    return false;
	if(conditional instanceof Boolean)
	    return (((Boolean)conditional).booleanValue() == targetValue);
	return false;
    }
}
