/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.zip.GZIPInputStream;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.io.Streams;
import net.sourceforge.javadpkg.io.impl.DataStreamSource;
import net.sourceforge.javadpkg.io.impl.DataStreamTarget;


/**
 * <p>
 * Performs some tests on the {@link Streams} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 09.05.2016 by Gerrit Hohl
 */
public class StreamsTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link Streams#compressGzip(DataSource, DataTarget, int)} method.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void testCompressGzip() {
		ByteArrayInputStream in;
		DataSource source;
		ByteArrayOutputStream out;
		DataTarget target;
		byte[] buffer;
		int len;
		
		
		in = new ByteArrayInputStream("Hello World!".getBytes());
		source = new DataStreamSource(in, "raw", false);
		out = new ByteArrayOutputStream();
		target = new DataStreamTarget(out, "gzip", false);


		// --- Parameters ---
		try {
			Streams.compressGzip(null, target, 9);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			Streams.compressGzip(source, null, 9);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			Streams.compressGzip(source, target, -2);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			Streams.compressGzip(source, target, 10);
			Assert.fail("Expected an exception, but none was thrown.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}


		// --- Compress successfully ---
		try {
			Streams.compressGzip(source, target, 9);
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		in = new ByteArrayInputStream(out.toByteArray());
		buffer = new byte[65536];
		try {
			try (GZIPInputStream gzipIn = new GZIPInputStream(in)) {
				len = gzipIn.read(buffer);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		Assert.assertEquals(12, len);
		Assert.assertEquals("Hello World!", new String(buffer, 0, len));
	}
	
	
}
