/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Template;
import net.sourceforge.javadpkg.Templates;
import net.sourceforge.javadpkg.TemplatesParser;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.control.Description;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.impl.TemplatesParserImpl;
import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * Performs some tests on the {@link TemplatesParserImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 06.05.2016 by Gerrit Hohl
 */
public class TemplatesParserImplTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link TemplatesParserImpl} class.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void test() {
		TemplatesParser parser;
		DataSource source;
		Context context;
		Templates templates;
		List<Warning> warnings;
		List<Template> templateList;
		Template template;
		Map<String, String> defaultValue;
		Map<String, List<String>> choices;
		Map<String, Description> descriptions;
		Description description;
		
		
		parser = new TemplatesParserImpl();
		source = createSource(Arrays.asList("Template: hostname", "Type: string", "Default: debian",
				"Description: unqualified hostname for this computer",
				" This is the name by which this computer will be known on the network. It",
				" has to be a unique name in your domain.", "", "Template: domain", "Type: string",
				"Description: domain for this computer", " This is the domain your computer is a member of. Typically it is",
				" something like \"mycompany.com\" or \"myuniversity.edu\"."), "templates");
		context = new ContextImpl();


		// --- Parameters ---
		try {
			parser.parseTemplates((DataSource) null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("Unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			parser.parseTemplates(source, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("Unexpected exception was thrown: " + e.getMessage());
			return;
		}


		// --- Parse successfully ---
		try {
			templates = parser.parseTemplates(source, context);
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("Unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(templates);
		templateList = templates.getTemplates();
		Assert.assertNotNull(templateList);
		Assert.assertEquals(2, templateList.size());

		template = templateList.get(0);
		Assert.assertEquals("hostname", template.getName());
		Assert.assertNotNull(template.getType());
		Assert.assertEquals("string", template.getType().getText());
		defaultValue = template.getDefaultValue();
		Assert.assertNotNull(defaultValue);
		Assert.assertEquals(1, defaultValue.size());
		Assert.assertEquals("debian", defaultValue.get(""));
		choices = template.getChoices();
		Assert.assertNull(choices);
		descriptions = template.getDescriptions();
		Assert.assertNotNull(descriptions);
		Assert.assertEquals(1, descriptions.size());
		description = descriptions.get("");
		Assert.assertNotNull(description);
		Assert.assertEquals("unqualified hostname for this computer\n"
				+ "This is the name by which this computer will be known on the network. It\n"
				+ "has to be a unique name in your domain.", description.getText());

		template = templateList.get(1);
		Assert.assertEquals("domain", template.getName());
		Assert.assertNotNull(template.getType());
		Assert.assertEquals("string", template.getType().getText());
		defaultValue = template.getDefaultValue();
		Assert.assertNotNull(defaultValue);
		Assert.assertEquals(0, defaultValue.size());
		choices = template.getChoices();
		Assert.assertNull(choices);
		descriptions = template.getDescriptions();
		Assert.assertNotNull(descriptions);
		Assert.assertEquals(1, descriptions.size());
		description = descriptions.get("");
		Assert.assertNotNull(description);
		Assert.assertEquals("domain for this computer\n" + "This is the domain your computer is a member of. Typically it is\n"
				+ "something like \"mycompany.com\" or \"myuniversity.edu\".", description.getText());
	}


}
