/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Script;
import net.sourceforge.javadpkg.ScriptVariableReplacer;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.impl.ScriptImpl;
import net.sourceforge.javadpkg.impl.ScriptVariableReplacerImpl;
import net.sourceforge.javadpkg.replace.ReplacementException;
import net.sourceforge.javadpkg.replace.Replacements;
import net.sourceforge.javadpkg.replace.ReplacementsMap;


/**
 * <p>
 * Performs some tests on the {@link ScriptVariableReplacerImpl}.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 02.05.2016 by Gerrit Hohl
 */
public class ScriptVariableReplacerImplTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the {@link ScriptVariableReplacerImpl}.
	 * </p>
	 */
	@Test
	public void test() {
		ScriptVariableReplacer replacer;
		Script script, scriptReplaced;
		Map<String, String> variables;
		Replacements replacements;
		Context context;
		List<Warning> warnings;


		replacer = new ScriptVariableReplacerImpl();

		// --- Parameters ---
		script = new ScriptImpl();
		replacements = new ReplacementsMap();
		context = new ContextImpl();
		try {
			replacer.replaceScriptVariables(null, replacements, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			replacer.replaceScriptVariables(script, null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			replacer.replaceScriptVariables(script, replacements, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		
		
		// --- Replace successfully ---
		script = new ScriptImpl(Arrays.asList("#!/bin/sh", "# --- Enable error detection. ---", "set -e", "",
				"# --- Set version of this package ---", "THISVERSION=\"${deb.version}\"", "",
				"echo Pre-installation script version $THISVERSION", "echo \"   Parameters                  : $@\"",
				"echo \"   Option                      : $1\"", "echo \"   Currently installed version : $2\"",
				"echo \"   Version of this package     : $THISVERSION\"", "", "case \"$1\" in", "    install)",
				"        echo \"   No tasks.\"", "        ;;", "    upgrade)", "        echo \"   No tasks.\"", "        ;;",
				"    *)", "        echo \"   Option $1 not supported by this script.\"", "        ;;", "esac", ""));
		variables = new HashMap<>();
		variables.put("deb.version", "1.0.0");
		replacements = new ReplacementsMap(variables);
		try {
			scriptReplaced = replacer.replaceScriptVariables(script, replacements, context);
		} catch (ReplacementException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertEquals("#!/bin/sh\n" + "# --- Enable error detection. ---\n" + "set -e\n" + "\n"
				+ "# --- Set version of this package ---\n" + "THISVERSION=\"${deb.version}\"\n" + "\n"
				+ "echo Pre-installation script version $THISVERSION\n" + "echo \"   Parameters                  : $@\"\n"
				+ "echo \"   Option                      : $1\"\n" + "echo \"   Currently installed version : $2\"\n"
				+ "echo \"   Version of this package     : $THISVERSION\"\n" + "\n" + "case \"$1\" in\n" + "    install)\n"
				+ "        echo \"   No tasks.\"\n" + "        ;;\n" + "    upgrade)\n" + "        echo \"   No tasks.\"\n"
				+ "        ;;\n" + "    *)\n" + "        echo \"   Option $1 not supported by this script.\"\n" + "        ;;\n"
				+ "esac\n", script.getText());
		Assert.assertNotNull(scriptReplaced);
		Assert.assertEquals("#!/bin/sh\n" + "# --- Enable error detection. ---\n" + "set -e\n" + "\n"
				+ "# --- Set version of this package ---\n" + "THISVERSION=\"1.0.0\"\n" + "\n"
				+ "echo Pre-installation script version $THISVERSION\n" + "echo \"   Parameters                  : $@\"\n"
				+ "echo \"   Option                      : $1\"\n" + "echo \"   Currently installed version : $2\"\n"
				+ "echo \"   Version of this package     : $THISVERSION\"\n" + "\n" + "case \"$1\" in\n" + "    install)\n"
				+ "        echo \"   No tasks.\"\n" + "        ;;\n" + "    upgrade)\n" + "        echo \"   No tasks.\"\n"
				+ "        ;;\n" + "    *)\n" + "        echo \"   Option $1 not supported by this script.\"\n" + "        ;;\n"
				+ "esac\n", scriptReplaced.getText());
	}


}
