/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.junit.Assert;
import org.junit.Ignore;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.ChangeLog;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Copyright;
import net.sourceforge.javadpkg.CopyrightConstants;
import net.sourceforge.javadpkg.DebianPackage;
import net.sourceforge.javadpkg.DebianPackageBuilder;
import net.sourceforge.javadpkg.DebianPackageBuilderFactory;
import net.sourceforge.javadpkg.DebianPackageConstants;
import net.sourceforge.javadpkg.DebianPackageParseHandler;
import net.sourceforge.javadpkg.DebianPackageParser;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.control.BinaryControl;
import net.sourceforge.javadpkg.control.impl.BinaryControlImpl;
import net.sourceforge.javadpkg.impl.ChangeLogImpl;
import net.sourceforge.javadpkg.impl.ChangeLogVersionEntryDetailImpl;
import net.sourceforge.javadpkg.impl.ChangeLogVersionEntryImpl;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.impl.CopyrightImpl;
import net.sourceforge.javadpkg.impl.CopyrightLicenseImpl;
import net.sourceforge.javadpkg.impl.DebianPackageBuilderFactoryImpl;
import net.sourceforge.javadpkg.impl.DebianPackageBuilderImpl;
import net.sourceforge.javadpkg.impl.DebianPackageParserImpl;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.io.FileMetaData;
import net.sourceforge.javadpkg.io.Streams;
import net.sourceforge.javadpkg.io.impl.DataStreamSource;
import net.sourceforge.javadpkg.io.impl.DataStreamTarget;
import net.sourceforge.javadpkg.io.impl.FileModeImpl;
import net.sourceforge.javadpkg.io.impl.FileOwnerImpl;


/**
 * <p>
 * Performs some tests on the {@link DebianPackageBuilderImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 27.04.2016 by Gerrit Hohl
 */
public class DebianPackageBuilderImplTest extends AbstractDpkgTest implements DebianPackageConstants {
	
	
	/** The parser for the Debian package. */
	private DebianPackageParser			parser;
	/** The factory for the builders for the Debian package. */
	private DebianPackageBuilderFactory	debianPackageBuilderFactory;
	/** The user home. */
	private File						userHome;
	
	
	@Override
	public void setUp() throws Exception {
		super.setUp();

		this.parser = new DebianPackageParserImpl();

		this.debianPackageBuilderFactory = new DebianPackageBuilderFactoryImpl();

		this.userHome = new File(System.getProperty("user.home"));
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DebianPackageBuilderImpl#buildDebianPackage(DataTarget, Context)}
	 * method by using illegal parameters.
	 * </p>
	 */
	@Test
	public void testParameters() {
		DebianPackageBuilder builder;
		ByteArrayOutputStream out;
		Context context;
		
		
		// --- Parameters ---
		builder = this.debianPackageBuilderFactory.createDebianPackageBuilder();
		out = new ByteArrayOutputStream();
		context = new ContextImpl();
		try {
			builder.buildDebianPackage(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		}
		try {
			try (DataTarget target = new DataStreamTarget(out, "test.deb", false)) {
				builder.buildDebianPackage(target, null);
			}
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		}
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DebianPackageBuilderImpl#buildDebianPackage(DataTarget, Context)}
	 * method without setting a control.
	 * </p>
	 */
	@Test
	public void testMissingControl() {
		DebianPackageBuilder builder;
		ByteArrayOutputStream out;
		Context context;
		
		
		// --- Missing control ---
		builder = this.debianPackageBuilderFactory.createDebianPackageBuilder();
		out = new ByteArrayOutputStream();
		context = new ContextImpl();
		try {
			try (DataTarget target = new DataStreamTarget(out, "test.deb", false)) {
				builder.buildDebianPackage(target, context);
			}
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalStateException e) {
			// --- Do nothing. Expected exception. ---
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		}
	}


	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DebianPackageBuilderImpl#buildDebianPackage(DataTarget, Context)}
	 * method without setting a copyright.
	 * </p>
	 */
	@Test
	public void testMissingCopyright() {
		DebianPackageBuilder builder;
		ByteArrayOutputStream out;
		Context context;
		BinaryControlImpl control;
		
		
		// --- Missing copyright ---
		builder = this.debianPackageBuilderFactory.createDebianPackageBuilder();
		out = new ByteArrayOutputStream();
		context = new ContextImpl();
		control = createBinaryControl("mypackage", "1.0.0", "net", "optional", "amd64", "oracle-8-jre (>= 1.8.0)",
				Long.valueOf(12345), "http://www.example.com/", "John Doe <j.doe@example.com>",
				"This is my first package.\n" + "I'm somewhat excited.\n" + ".\n" + "This will be a great package.\n" + ".");
		builder.setControl(control);
		try {
			try (DataTarget target = new DataStreamTarget(out, "test.deb", false)) {
				builder.buildDebianPackage(target, context);
			}
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalStateException e) {
			// --- Do nothing. Expected exception. ---
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		}
	}


	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DebianPackageBuilderImpl#buildDebianPackage(DataTarget, Context)}
	 * method without setting a change log.
	 * </p>
	 */
	@Test
	public void testMissingChangeLog() {
		DebianPackageBuilder builder;
		ByteArrayOutputStream out;
		Context context;
		BinaryControlImpl control;
		
		
		// --- Missing change log ---
		builder = this.debianPackageBuilderFactory.createDebianPackageBuilder();
		out = new ByteArrayOutputStream();
		context = new ContextImpl();
		control = createBinaryControl("mypackage", "1.0.0", "net", "optional", "amd64", "oracle-8-jre (>= 1.8.0)",
				Long.valueOf(12345), "http://www.example.com/", "John Doe <j.doe@example.com>",
				"This is my first package.\n" + "I'm somewhat excited.\n" + ".\n" + "This will be a great package.\n" + ".");
		builder.setControl(control);
		builder.setCopyright(this.createCopyright());
		try {
			try (DataTarget target = new DataStreamTarget(out, "test.deb", false)) {
				builder.buildDebianPackage(target, context);
			}
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalStateException e) {
			// --- Do nothing. Expected exception. ---
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		}
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DebianPackageBuilderImpl#buildDebianPackage(DataTarget, Context)}
	 * method without any files.
	 * </p>
	 */
	@Test
	public void testNoFiles() {
		DebianPackageBuilder builder;
		ByteArrayOutputStream out;
		Context context;
		List<Warning> warnings;
		BinaryControlImpl control;
		DebianPackageParseCollectHandler handler;
		List<FileMetaData> files;


		// --- No files ---
		builder = this.debianPackageBuilderFactory.createDebianPackageBuilder();
		out = new ByteArrayOutputStream();
		context = new ContextImpl();
		control = createBinaryControl("mypackage", "1.0.0", "net", "optional", "amd64", "oracle-8-jre (>= 1.8.0)",
				Long.valueOf(12345), "http://www.example.com/", "John Doe <j.doe@example.com>",
				"This is my first package.\n" + "I'm somewhat excited.\n" + ".\n" + "This will be a great package.\n" + ".");
		builder.setControl(control);
		builder.setCopyright(this.createCopyright());
		builder.setChangeLog(this.createChangeLog(control));
		try {
			try (DataTarget target = new DataStreamTarget(out, "test.deb", false)) {
				builder.buildDebianPackage(target, context);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		handler = new DebianPackageParseCollectHandler();
		this.parseDebianPackage(out.toByteArray(), handler);
		handler.printFiles(System.out);
		files = handler.getFiles();
		Assert.assertEquals(5, files.size());
		Assert.assertEquals("/", files.get(0).getAbsolutePath());
		Assert.assertEquals("/usr/", files.get(1).getAbsolutePath());
		Assert.assertEquals("/usr/share/", files.get(2).getAbsolutePath());
		Assert.assertEquals("/usr/share/doc/", files.get(3).getAbsolutePath());
		Assert.assertEquals("/usr/share/doc/mypackage/", files.get(4).getAbsolutePath());
	}


	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DebianPackageBuilderImpl#buildDebianPackage(DataTarget, Context)}
	 * method with some files.
	 * </p>
	 */
	@Test
	public void testSomeFiles() {
		DebianPackageBuilder builder;
		ByteArrayOutputStream out;
		Context context;
		List<Warning> warnings;
		BinaryControlImpl control;
		DebianPackageParseCollectHandler handler;
		List<FileMetaData> files;


		// --- Some files ---
		builder = this.debianPackageBuilderFactory.createDebianPackageBuilder();
		out = new ByteArrayOutputStream();
		context = new ContextImpl();
		control = createBinaryControl("mypackage", "1.0.0", "net", "optional", "amd64", "oracle-8-jre (>= 1.8.0)", null,
				"http://www.example.com/", "John Doe <j.doe@example.com>",
				"This is my first package.\n" + "I'm somewhat excited.\n" + ".\n" + "This will be a great package.\n" + ".");
		builder.setControl(control);
		builder.addDataDirectory("/usr");
		builder.addDataDirectory("/usr/local");
		builder.addDataDirectory("/usr/local/myprogram");
		builder.addDataFile(createSource(1000, "myprogram"), "/usr/local/myprogram/myprogram",
				new FileOwnerImpl(ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME), new FileModeImpl(SCRIPT_MODE));
		builder.addDataSymLink("/usr/local/myprogram/mylink", "/usr/local/myprogram",
				new FileOwnerImpl(ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME), new FileModeImpl(SCRIPT_MODE));
		builder.addDataDirectory("/usr/local/myprogram/libs");
		builder.addDataFile(createSource(2000, "first.jar"), "/usr/local/myprogram/libs/first.jar");
		builder.addDataFile(createSource(50000, "second.jar"), "/usr/local/myprogram/libs/second.jar");
		builder.addDataFile(createSource(100000, "third.jar"), "/usr/local/myprogram/libs/third.jar");
		builder.addDataFile(createSource(25000, "fourth.jar"), "/usr/local/myprogram/libs/fourth.jar");
		builder.setCopyright(this.createCopyright());
		builder.setChangeLog(this.createChangeLog(control));
		try {
			try (DataTarget target = new DataStreamTarget(out, "test.deb", false)) {
				builder.buildDebianPackage(target, context);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		handler = new DebianPackageParseCollectHandler();
		this.parseDebianPackage(out.toByteArray(), handler);
		handler.printFiles(System.out);
		files = handler.getFiles();
		Assert.assertEquals(14, files.size());
		Assert.assertEquals("/", files.get(0).getAbsolutePath());
		Assert.assertEquals("/usr/", files.get(1).getAbsolutePath());
		Assert.assertEquals("/usr/local/", files.get(2).getAbsolutePath());
		Assert.assertEquals("/usr/local/myprogram/", files.get(3).getAbsolutePath());
		Assert.assertEquals("/usr/local/myprogram/libs/", files.get(4).getAbsolutePath());
		Assert.assertEquals("/usr/local/myprogram/libs/first.jar", files.get(5).getAbsolutePath());
		Assert.assertEquals("/usr/local/myprogram/libs/fourth.jar", files.get(6).getAbsolutePath());
		Assert.assertEquals("/usr/local/myprogram/libs/second.jar", files.get(7).getAbsolutePath());
		Assert.assertEquals("/usr/local/myprogram/libs/third.jar", files.get(8).getAbsolutePath());
		Assert.assertEquals("/usr/local/myprogram/mylink", files.get(9).getAbsolutePath());
		Assert.assertEquals("/usr/local/myprogram/myprogram", files.get(10).getAbsolutePath());
		Assert.assertEquals("/usr/share/", files.get(11).getAbsolutePath());
		Assert.assertEquals("/usr/share/doc/", files.get(12).getAbsolutePath());
		Assert.assertEquals("/usr/share/doc/mypackage/", files.get(13).getAbsolutePath());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link DebianPackageBuilderImpl#buildDebianPackage(DataTarget, Context)}
	 * method with some files and writes it on the desktop.
	 * </p>
	 */
	@Test
	@Ignore
	public void testWritePackage() {
		DebianPackageBuilder builder;
		ByteArrayOutputStream out;
		Context context;
		List<Warning> warnings;
		BinaryControlImpl control;


		// --- Some files ---
		builder = this.debianPackageBuilderFactory.createDebianPackageBuilder();
		out = new ByteArrayOutputStream();
		context = new ContextImpl();
		control = createBinaryControl("mypackage", "1.0.0", "net", "optional", "amd64", "oracle-8-jre (>= 1.8.0)", null,
				"http://www.example.com/", "John Doe <j.doe@example.com>",
				"This is my first package.\n" + "I'm somewhat excited.\n" + ".\n" + "This will be a great package.\n" + ".");
		builder.setControl(control);
		builder.addDataDirectory("/usr");
		builder.addDataDirectory("/usr/local");
		builder.addDataDirectory("/usr/local/myprogram");
		builder.addDataFile(createSource(1000, "myprogram"), "/usr/local/myprogram/myprogram",
				new FileOwnerImpl(ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME), new FileModeImpl(SCRIPT_MODE));
		builder.addDataSymLink("/usr/local/mylink", "/usr/local/myprogram",
				new FileOwnerImpl(ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME), new FileModeImpl(SCRIPT_MODE));
		builder.addDataDirectory("/usr/local/myprogram/libs");
		builder.addDataFile(createSource(2000, "first.jar"), "/usr/local/myprogram/libs/first.jar");
		builder.addDataFile(createSource(50000, "second.jar"), "/usr/local/myprogram/libs/second.jar");
		builder.addDataFile(createSource(100000, "third.jar"), "/usr/local/myprogram/libs/third.jar");
		builder.addDataFile(createSource(25000, "fourth.jar"), "/usr/local/myprogram/libs/fourth.jar");
		builder.setCopyright(this.createCopyright());
		builder.setChangeLog(this.createChangeLog(control));
		try {
			try (DataTarget target = new DataStreamTarget(out, "test.deb", false)) {
				builder.buildDebianPackage(target, context);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		} catch (BuildException e) {
			e.printStackTrace();
			Assert.fail("Couldn't create Debian package: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		try {
			try (OutputStream fos = Streams
					.createBufferedFileOutputStream(new File(this.userHome, "Desktop\\mypackage-1.0.0-amd64.deb"))) {
				Streams.copy(new ByteArrayInputStream(out.toByteArray()), fos);
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
	}


	/**
	 * <p>
	 * Creates a copyright.
	 * </p>
	 *
	 * @return The copyright.
	 */
	private Copyright createCopyright() {
		CopyrightImpl copyright;
		CopyrightLicenseImpl license;
		
		
		copyright = new CopyrightImpl();
		copyright.setFormat(CopyrightConstants.FORMAT_V1_0);
		copyright.setCopyright("(c) 2015, John Doe");
		license = new CopyrightLicenseImpl();
		license.setName("MyLicense");
		license.setText("This is my license...");
		copyright.setLicense(license);
		return copyright;
	}


	/**
	 * <p>
	 * Creates a change log.
	 * </p>
	 *
	 * @param control
	 *            The binary control.
	 * @return The change log.
	 */
	private ChangeLog createChangeLog(BinaryControl control) {
		ChangeLogImpl changeLog;
		ChangeLogVersionEntryImpl entry;
		ChangeLogVersionEntryDetailImpl detail;


		changeLog = new ChangeLogImpl();
		entry = new ChangeLogVersionEntryImpl();
		entry.setPackageName(control.getPackage());
		entry.setVersion(control.getVersion());
		entry.setDistributions(Arrays.asList("lucid"));
		entry.setUrgency(createChangeLogUrgency("low"));
		detail = new ChangeLogVersionEntryDetailImpl("First version of the package.");
		entry.addDetail(detail);
		entry.setMaintainer(control.getMaintainer());
		entry.setDate(new Date(1462778238028L));
		changeLog.addEntry(entry);
		return changeLog;
	}


	/**
	 * <p>
	 * Parses the Debian package from the specified array.
	 * </p>
	 *
	 * @param data
	 *            The array.
	 * @param handler
	 *            The handler.
	 * @return The Debian package.
	 */
	private DebianPackage parseDebianPackage(byte[] data, DebianPackageParseHandler handler) {
		DebianPackage debianPackage;
		Context context;


		context = new ContextImpl();
		try {
			try (DataSource source = new DataStreamSource(new ByteArrayInputStream(data), "control", false)) {
				debianPackage = this.parser.parseDebianPackage(source, handler, context);
			}
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("The Debian package is invalid: " + e.getMessage());
			return null;
		}
		return debianPackage;
	}


}
