/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.field.impl;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.field.Field;
import net.sourceforge.javadpkg.field.FieldParser;
import net.sourceforge.javadpkg.field.impl.FieldParserImpl;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * Performs some tests on the {@link FieldParserImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 28.04.2016 by Gerrit Hohl
 */
public class FieldParserImplTest extends AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FieldParserImpl#parseFieldsAsList(DataSource, Context)} method.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void testParseFieldsAsList() {
		FieldParser parser;
		Context context;
		DataSource source;
		List<Field> fields;
		List<Warning> warnings;
		Field field;


		context = new ContextImpl();
		source = createSource(Arrays.asList("Package: mypackage", "Version: 1.0.0", "Section: net", "Priority: optional",
				"Architecture: amd64", "Depends: oracle-8-jre (>= 1.8.0)", "Installed-Size: 12345",
				"Homepage: http://www.example.com/", "Maintainer: John Doe <j.doe@example.com>",
				"Description: This is my first package.", " I'm somewhat excited.", " .", " This will be a great package.",
				" .", ""), "control");
		parser = new FieldParserImpl(false, false, false);
		
		
		// --- Parameters ---
		try {
			fields = parser.parseFieldsAsList(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			fields = parser.parseFieldsAsList(source, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}

		// --- Parse successfully ---
		try {
			fields = parser.parseFieldsAsList(source, context);
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(fields);
		Assert.assertEquals(10, fields.size());

		field = fields.get(0);
		Assert.assertNotNull(field);
		Assert.assertEquals("Package", field.getName());
		Assert.assertEquals("mypackage", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(1);
		Assert.assertNotNull(field);
		Assert.assertEquals("Version", field.getName());
		Assert.assertEquals("1.0.0", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(2);
		Assert.assertNotNull(field);
		Assert.assertEquals("Section", field.getName());
		Assert.assertEquals("net", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(3);
		Assert.assertNotNull(field);
		Assert.assertEquals("Priority", field.getName());
		Assert.assertEquals("optional", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(4);
		Assert.assertNotNull(field);
		Assert.assertEquals("Architecture", field.getName());
		Assert.assertEquals("amd64", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(5);
		Assert.assertNotNull(field);
		Assert.assertEquals("Depends", field.getName());
		Assert.assertEquals("oracle-8-jre (>= 1.8.0)", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(6);
		Assert.assertNotNull(field);
		Assert.assertEquals("Installed-Size", field.getName());
		Assert.assertEquals("12345", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(7);
		Assert.assertNotNull(field);
		Assert.assertEquals("Homepage", field.getName());
		Assert.assertEquals("http://www.example.com/", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(8);
		Assert.assertNotNull(field);
		Assert.assertEquals("Maintainer", field.getName());
		Assert.assertEquals("John Doe <j.doe@example.com>", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(9);
		Assert.assertNotNull(field);
		Assert.assertEquals("Description", field.getName());
		Assert.assertEquals(
				"This is my first package.\n" + "I'm somewhat excited.\n" + ".\n" + "This will be a great package.\n" + ".",
				field.getValue());
		Assert.assertFalse(field.isEmpty());


		// --- Allow empty lines, but don't return them ---
		parser = new FieldParserImpl(true, false, false);
		source = createSource(Arrays.asList("Field1: value1", "", "Field2: value2", ""), "fields");
		try {
			fields = parser.parseFieldsAsList(source, context);
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(fields);
		Assert.assertEquals(2, fields.size());

		field = fields.get(0);
		Assert.assertNotNull(field);
		Assert.assertEquals("Field1", field.getName());
		Assert.assertEquals("value1", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(1);
		Assert.assertNotNull(field);
		Assert.assertEquals("Field2", field.getName());
		Assert.assertEquals("value2", field.getValue());
		Assert.assertFalse(field.isEmpty());


		// --- Allow empty lines and return them ---
		parser = new FieldParserImpl(true, true, false);
		source = createSource(Arrays.asList("Field1: value1", "", "Field2: value2", ""), "fields");
		try {
			fields = parser.parseFieldsAsList(source, context);
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(fields);
		Assert.assertEquals(4, fields.size());

		field = fields.get(0);
		Assert.assertNotNull(field);
		Assert.assertEquals("Field1", field.getName());
		Assert.assertEquals("value1", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(1);
		Assert.assertNotNull(field);
		Assert.assertNull(field.getName());
		Assert.assertNull(field.getValue());
		Assert.assertTrue(field.isEmpty());

		field = fields.get(2);
		Assert.assertNotNull(field);
		Assert.assertEquals("Field2", field.getName());
		Assert.assertEquals("value2", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get(3);
		Assert.assertNotNull(field);
		Assert.assertNull(field.getName());
		Assert.assertNull(field.getValue());
		Assert.assertTrue(field.isEmpty());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link FieldParserImpl#parseFieldsAsMap(DataSource, Context)} method.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void testParseFieldsAsMap() {
		FieldParser parser;
		Context context;
		DataSource source;
		Map<String, Field> fields;
		List<Warning> warnings;
		Field field;


		context = new ContextImpl();
		source = createSource(Arrays.asList("Package: mypackage", "Version: 1.0.0", "Section: net", "Priority: optional",
				"Architecture: amd64", "Depends: oracle-8-jre (>=1.8.0)", "Installed-Size: 12345",
				"Homepage: http://www.example.com/", "Maintainer: John Doe <j.doe@example.com>",
				"Description: This is my first package.", " I'm somewhat excited.", " .", " This will be a great package.",
				" .", ""), "control");
		parser = new FieldParserImpl(false, false, false);
		
		
		// --- Parameters ---
		try {
			fields = parser.parseFieldsAsMap(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			fields = parser.parseFieldsAsMap(source, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}

		// --- Parse successfully ---
		try {
			fields = parser.parseFieldsAsMap(source, context);
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(fields);
		Assert.assertEquals(10, fields.size());

		field = fields.get("package");
		Assert.assertNotNull(field);
		Assert.assertEquals("Package", field.getName());
		Assert.assertEquals("mypackage", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("version");
		Assert.assertNotNull(field);
		Assert.assertEquals("Version", field.getName());
		Assert.assertEquals("1.0.0", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("section");
		Assert.assertNotNull(field);
		Assert.assertEquals("Section", field.getName());
		Assert.assertEquals("net", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("priority");
		Assert.assertNotNull(field);
		Assert.assertEquals("Priority", field.getName());
		Assert.assertEquals("optional", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("architecture");
		Assert.assertNotNull(field);
		Assert.assertEquals("Architecture", field.getName());
		Assert.assertEquals("amd64", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("depends");
		Assert.assertNotNull(field);
		Assert.assertEquals("Depends", field.getName());
		Assert.assertEquals("oracle-8-jre (>=1.8.0)", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("installed-size");
		Assert.assertNotNull(field);
		Assert.assertEquals("Installed-Size", field.getName());
		Assert.assertEquals("12345", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("homepage");
		Assert.assertNotNull(field);
		Assert.assertEquals("Homepage", field.getName());
		Assert.assertEquals("http://www.example.com/", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("maintainer");
		Assert.assertNotNull(field);
		Assert.assertEquals("Maintainer", field.getName());
		Assert.assertEquals("John Doe <j.doe@example.com>", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("description");
		Assert.assertNotNull(field);
		Assert.assertEquals("Description", field.getName());
		Assert.assertEquals(
				"This is my first package.\n" + "I'm somewhat excited.\n" + ".\n" + "This will be a great package.\n" + ".",
				field.getValue());
		Assert.assertFalse(field.isEmpty());
		
		
		// --- Allow empty lines, but don't return them ---
		parser = new FieldParserImpl(true, false, false);
		source = createSource(Arrays.asList("Field1: value1", "", "Field2: value2", ""), "fields");
		try {
			fields = parser.parseFieldsAsMap(source, context);
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(fields);
		Assert.assertEquals(2, fields.size());

		field = fields.get("field1");
		Assert.assertNotNull(field);
		Assert.assertEquals("Field1", field.getName());
		Assert.assertEquals("value1", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("field2");
		Assert.assertNotNull(field);
		Assert.assertEquals("Field2", field.getName());
		Assert.assertEquals("value2", field.getValue());
		Assert.assertFalse(field.isEmpty());


		// --- Allow empty lines and return them ---
		parser = new FieldParserImpl(true, true, false);
		source = createSource(Arrays.asList("Field1: value1", "", "Field2: value2", ""), "fields");
		try {
			fields = parser.parseFieldsAsMap(source, context);
		} catch (ParseException | IOException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(fields);
		Assert.assertEquals(2, fields.size());

		field = fields.get("field1");
		Assert.assertNotNull(field);
		Assert.assertEquals("Field1", field.getName());
		Assert.assertEquals("value1", field.getValue());
		Assert.assertFalse(field.isEmpty());

		field = fields.get("field2");
		Assert.assertNotNull(field);
		Assert.assertEquals("Field2", field.getName());
		Assert.assertEquals("value2", field.getValue());
		Assert.assertFalse(field.isEmpty());
	}
	
	
}
