/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2018 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.control.PackageVersion;
import net.sourceforge.javadpkg.impl.ContextImpl;


/**
 * <p>
 * Performs some tests on the {@link PackageVersionParserImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 23.01.2018 by Gerrit Hohl
 */
public class PackageVersionParserImplTest extends AbstractDpkgTest {


	/**
	 * <p>
	 * Performs some tests on the {@link PackageVersionParserImpl} class.
	 * </p>
	 */
	@Test
	public void test() {
		PackageVersionParserImpl parser;
		ContextImpl context;
		PackageVersion version;


		parser = new PackageVersionParserImpl();


		// --- Parameters ---
		context = new ContextImpl();
		try {
			parser.parsePackageVersion(null, context);
			Assert.fail("Expected an exception, but the method didn't throw any.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Method threw unexpected exception: " + e.getMessage());
			return;
		}
		try {
			parser.parsePackageVersion("", null);
			Assert.fail("Expected an exception, but the method didn't throw any.");
			return;
		} catch (IllegalArgumentException e) {
			// --- Do nothing. Expected exception. ---
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Method threw unexpected exception: " + e.getMessage());
			return;
		}


		// --- Valid version ---
		context = new ContextImpl();
		try {
			version = parser.parsePackageVersion("1.2.3", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Method threw unexpected exception: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(version);
		Assert.assertEquals("1.2.3", version.getText());
		Assert.assertNull(version.getEpoch());
		Assert.assertEquals("1.2.3", version.getUpstreamVersion());
		Assert.assertNull(version.getDebianRevision());


		// --- Valid epoch and version ---
		context = new ContextImpl();
		try {
			version = parser.parsePackageVersion("1:2.3.4", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Method threw unexpected exception: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(version);
		Assert.assertEquals("1:2.3.4", version.getText());
		Assert.assertEquals("1", version.getEpoch());
		Assert.assertEquals("2.3.4", version.getUpstreamVersion());
		Assert.assertNull(version.getDebianRevision());

		// --- Valid version and revision ---
		context = new ContextImpl();
		try {
			version = parser.parsePackageVersion("1.2.3-4", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Method threw unexpected exception: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(version);
		Assert.assertEquals("1.2.3-4", version.getText());
		Assert.assertNull(version.getEpoch());
		Assert.assertEquals("1.2.3", version.getUpstreamVersion());
		Assert.assertEquals("4", version.getDebianRevision());


		// --- Valid epoch, version and revision ---
		context = new ContextImpl();
		try {
			version = parser.parsePackageVersion("1:2.3.4-5", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Method threw unexpected exception: " + e.getMessage());
			return;
		}
		Assert.assertNotNull(version);
		Assert.assertEquals("1:2.3.4-5", version.getText());
		Assert.assertEquals("1", version.getEpoch());
		Assert.assertEquals("2.3.4", version.getUpstreamVersion());
		Assert.assertEquals("5", version.getDebianRevision());
	}


}
