/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import java.util.List;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.control.PackageDependency;
import net.sourceforge.javadpkg.control.PackageDependency.Condition;
import net.sourceforge.javadpkg.control.PackageDependencyParser;
import net.sourceforge.javadpkg.control.PackageNameParser;
import net.sourceforge.javadpkg.control.PackageVersionParser;
import net.sourceforge.javadpkg.control.PackageVersionRelationOperatorParser;
import net.sourceforge.javadpkg.impl.ContextImpl;


/**
 * <p>
 * Performs some tests on the {@link PackageDependencyParserImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 29.04.2016 by Gerrit Hohl
 * @version <b>1.0</b>, 23.01.2018 by Gerrit Hohl
 */
public class PackageDependencyParserImplTest extends AbstractDpkgTest {
	
	
	/** The parser for the package name. */
	private PackageNameParser						packageNameParser;
	/** The parser for the version relation operator. */
	private PackageVersionRelationOperatorParser	packageVersionRelationOperatorParser;
	/** The parser for the version. */
	private PackageVersionParser					packageVersionParser;


	@Override
	public void setUp() throws Exception {
		super.setUp();

		this.packageNameParser = new PackageNameParserImpl();
		this.packageVersionRelationOperatorParser = new PackageVersionRelationOperatorParserImpl();
		this.packageVersionParser = new PackageVersionParserImpl();
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link PackageDependencyParserImpl#parsePackageDependency(String, Context)}
	 * method.
	 * </p>
	 */
	@Test
	public void testParsePackageDependency() {
		PackageDependencyParser parser;
		Context context;
		PackageDependency dependency, dep;
		List<Warning> warnings;


		parser = new PackageDependencyParserImpl(this.packageNameParser, this.packageVersionRelationOperatorParser,
				this.packageVersionParser);
		context = new ContextImpl();


		// --- Parameters ---
		try {
			parser.parsePackageDependency(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			parser.parsePackageDependency("libc6 (>= 2.14)", null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		
		
		// --- Parse successfully ---
		try {
			dependency = parser.parsePackageDependency("libc6 (>= 2.14)", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(dependency);
		Assert.assertFalse(dependency.isCondition());
		Assert.assertNull(dependency.getCondition());
		Assert.assertNotNull(dependency.getConditionPackageDependencies());
		Assert.assertEquals(0, dependency.getConditionPackageDependencies().size());
		Assert.assertNotNull(dependency.getPackage());
		Assert.assertEquals("libc6", dependency.getPackage().getName());
		Assert.assertNotNull(dependency.getRelationOperator());
		Assert.assertEquals(">=", dependency.getRelationOperator().getText());
		Assert.assertNotNull(dependency.getVersion());
		Assert.assertNull(dependency.getVersion().getEpoch());
		Assert.assertEquals("2.14", dependency.getVersion().getUpstreamVersion());
		Assert.assertNull(dependency.getVersion().getDebianRevision());
		
		
		// --- Parse successfully (ticket #3 - Can't parse "libgcrypt20 (>=1.6.0-0)" ---
		try {
			dependency = parser.parsePackageDependency("libgcrypt20 (>=1.6.0-0)", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(dependency);
		Assert.assertFalse(dependency.isCondition());
		Assert.assertNull(dependency.getCondition());
		Assert.assertNotNull(dependency.getConditionPackageDependencies());
		Assert.assertEquals(0, dependency.getConditionPackageDependencies().size());
		Assert.assertNotNull(dependency.getPackage());
		Assert.assertEquals("libgcrypt20", dependency.getPackage().getName());
		Assert.assertNotNull(dependency.getRelationOperator());
		Assert.assertEquals(">=", dependency.getRelationOperator().getText());
		Assert.assertNotNull(dependency.getVersion());
		Assert.assertNull(dependency.getVersion().getEpoch());
		Assert.assertEquals("1.6.0", dependency.getVersion().getUpstreamVersion());
		Assert.assertEquals("0", dependency.getVersion().getDebianRevision());


		// --- Parse successfully (ticket #5 - Can't parse "util-linux (2.27.1-6ubuntu3.3)" ---
		try {
			dependency = parser.parsePackageDependency("util-linux (2.27.1-6ubuntu3.3)", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(dependency);
		Assert.assertFalse(dependency.isCondition());
		Assert.assertNull(dependency.getCondition());
		Assert.assertNotNull(dependency.getConditionPackageDependencies());
		Assert.assertEquals(0, dependency.getConditionPackageDependencies().size());
		Assert.assertNotNull(dependency.getPackage());
		Assert.assertEquals("util-linux", dependency.getPackage().getName());
		Assert.assertNull(dependency.getRelationOperator());
		Assert.assertNotNull(dependency.getVersion());
		Assert.assertNull(dependency.getVersion().getEpoch());
		Assert.assertEquals("2.27.1", dependency.getVersion().getUpstreamVersion());
		Assert.assertEquals("6ubuntu3.3", dependency.getVersion().getDebianRevision());
		
		
		// --- Parse successfully (Condition) ---
		try {
			dependency = parser.parsePackageDependency("debconf (>= 0.5) | debconf-2.0", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(dependency);
		Assert.assertTrue(dependency.isCondition());
		Assert.assertEquals(Condition.OR, dependency.getCondition());
		Assert.assertNotNull(dependency.getConditionPackageDependencies());
		Assert.assertEquals(2, dependency.getConditionPackageDependencies().size());
		// --- First argument ---
		dep = dependency.getConditionPackageDependencies().get(0);
		Assert.assertFalse(dep.isCondition());
		Assert.assertNull(dep.getCondition());
		Assert.assertNotNull(dep.getConditionPackageDependencies());
		Assert.assertEquals(0, dep.getConditionPackageDependencies().size());
		Assert.assertNotNull(dep.getPackage());
		Assert.assertEquals("debconf", dep.getPackage().getName());
		Assert.assertNotNull(dep.getRelationOperator());
		Assert.assertEquals(">=", dep.getRelationOperator().getText());
		Assert.assertNotNull(dep.getVersion());
		Assert.assertNull(dep.getVersion().getEpoch());
		Assert.assertEquals("0.5", dep.getVersion().getUpstreamVersion());
		Assert.assertNull(dep.getVersion().getDebianRevision());
		// --- Second argument ---
		dep = dependency.getConditionPackageDependencies().get(1);
		Assert.assertFalse(dep.isCondition());
		Assert.assertNull(dep.getCondition());
		Assert.assertNotNull(dep.getConditionPackageDependencies());
		Assert.assertEquals(0, dep.getConditionPackageDependencies().size());
		Assert.assertNotNull(dep.getPackage());
		Assert.assertEquals("debconf-2.0", dep.getPackage().getName());
		Assert.assertNull(dep.getRelationOperator());
		Assert.assertNull(dep.getVersion());
		// --- Rest of dependency ---
		Assert.assertNull(dependency.getPackage());
		Assert.assertNull(dependency.getRelationOperator());
		Assert.assertNull(dependency.getVersion());
	}
	
	
	/**
	 * <p>
	 * Performs some tests on the
	 * {@link PackageDependencyParserImpl#parsePackageDependencies(String, Context)}
	 * method.
	 * </p>
	 */
	@Test
	public void testParsePackageDependencies() {
		PackageDependencyParser parser;
		Context context;
		List<PackageDependency> dependencies;
		PackageDependency dependency, dep;
		List<Warning> warnings;


		parser = new PackageDependencyParserImpl(this.packageNameParser, this.packageVersionRelationOperatorParser,
				this.packageVersionParser);
		context = new ContextImpl();


		// --- Parameters ---
		try {
			parser.parsePackageDependencies(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			parser.parsePackageDependencies("libc6 (>= 2.14)", null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		
		
		// --- Parse successfully ---
		try {
			dependencies = parser.parsePackageDependencies("libc6 (>= 2.14)", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(1, dependencies.size());
		dependency = dependencies.get(0);
		Assert.assertNotNull(dependency);
		Assert.assertFalse(dependency.isCondition());
		Assert.assertNull(dependency.getCondition());
		Assert.assertNotNull(dependency.getConditionPackageDependencies());
		Assert.assertEquals(0, dependency.getConditionPackageDependencies().size());
		Assert.assertNotNull(dependency.getPackage());
		Assert.assertEquals("libc6", dependency.getPackage().getName());
		Assert.assertNotNull(dependency.getRelationOperator());
		Assert.assertEquals(">=", dependency.getRelationOperator().getText());
		Assert.assertNotNull(dependency.getVersion());
		Assert.assertNull(dependency.getVersion().getEpoch());
		Assert.assertEquals("2.14", dependency.getVersion().getUpstreamVersion());
		Assert.assertNull(dependency.getVersion().getDebianRevision());


		// --- Parse successfully (Condition) ---
		try {
			dependencies = parser.parsePackageDependencies("libc6 (>= 2.14), debconf (>= 0.5) | debconf-2.0", context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(2, dependencies.size());
		
		dependency = dependencies.get(0);
		Assert.assertNotNull(dependency);
		Assert.assertFalse(dependency.isCondition());
		Assert.assertNull(dependency.getCondition());
		Assert.assertNotNull(dependency.getConditionPackageDependencies());
		Assert.assertEquals(0, dependency.getConditionPackageDependencies().size());
		Assert.assertNotNull(dependency.getPackage());
		Assert.assertEquals("libc6", dependency.getPackage().getName());
		Assert.assertNotNull(dependency.getRelationOperator());
		Assert.assertEquals(">=", dependency.getRelationOperator().getText());
		Assert.assertNotNull(dependency.getVersion());
		Assert.assertNull(dependency.getVersion().getEpoch());
		Assert.assertEquals("2.14", dependency.getVersion().getUpstreamVersion());
		Assert.assertNull(dependency.getVersion().getDebianRevision());
		
		dependency = dependencies.get(1);
		Assert.assertNotNull(dependency);
		Assert.assertTrue(dependency.isCondition());
		Assert.assertEquals(Condition.OR, dependency.getCondition());
		Assert.assertNotNull(dependency.getConditionPackageDependencies());
		Assert.assertEquals(2, dependency.getConditionPackageDependencies().size());
		// --- First argument ---
		dep = dependency.getConditionPackageDependencies().get(0);
		Assert.assertFalse(dep.isCondition());
		Assert.assertNull(dep.getCondition());
		Assert.assertNotNull(dep.getConditionPackageDependencies());
		Assert.assertEquals(0, dep.getConditionPackageDependencies().size());
		Assert.assertNotNull(dep.getPackage());
		Assert.assertEquals("debconf", dep.getPackage().getName());
		Assert.assertNotNull(dep.getRelationOperator());
		Assert.assertEquals(">=", dep.getRelationOperator().getText());
		Assert.assertNotNull(dep.getVersion());
		Assert.assertNull(dep.getVersion().getEpoch());
		Assert.assertEquals("0.5", dep.getVersion().getUpstreamVersion());
		Assert.assertNull(dep.getVersion().getDebianRevision());
		// --- Second argument ---
		dep = dependency.getConditionPackageDependencies().get(1);
		Assert.assertFalse(dep.isCondition());
		Assert.assertNull(dep.getCondition());
		Assert.assertNotNull(dep.getConditionPackageDependencies());
		Assert.assertEquals(0, dep.getConditionPackageDependencies().size());
		Assert.assertNotNull(dep.getPackage());
		Assert.assertEquals("debconf-2.0", dep.getPackage().getName());
		Assert.assertNull(dep.getRelationOperator());
		Assert.assertNull(dep.getVersion());
		// --- Rest of dependency ---
		Assert.assertNull(dependency.getPackage());
		Assert.assertNull(dependency.getRelationOperator());
		Assert.assertNull(dependency.getVersion());
	}


}
