/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.control.impl;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import org.junit.Assert;
import org.junit.Test;

import net.sourceforge.javadpkg.AbstractDpkgTest;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.control.BinaryControl;
import net.sourceforge.javadpkg.control.Control;
import net.sourceforge.javadpkg.control.ControlParser;
import net.sourceforge.javadpkg.control.PackageDependency;
import net.sourceforge.javadpkg.control.impl.ControlParserImpl;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * Performs some tests on the {@link ControlParserImpl} class.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 28.04.2016 by Gerrit Hohl
 */
public class ControlParserImplTest extends AbstractDpkgTest {


	/**
	 * <p>
	 * Performs some tests on the {@link ControlParserImpl} class.
	 * </p>
	 */
	@SuppressWarnings("resource")
	@Test
	public void test() {
		ControlParser parser;
		DataSource source;
		Context context;
		Control control;
		List<Warning> warnings;
		BinaryControl binaryControl;
		List<PackageDependency> dependencies;
		PackageDependency dependency;


		parser = new ControlParserImpl();
		source = createSource(Arrays.asList("Package: mypackage", "Version: 1.0.0", "Section: net", "Priority: optional",
				"Architecture: amd64", "Depends: oracle-8-jre (>= 1.8.0)", "Installed-Size: 12345",
				"Homepage: http://www.example.com/", "Maintainer: John Doe <j.doe@example.com>",
				"Description: This is my first package.", " I'm somewhat excited.", " .", " This will be a great package.",
				" .\n"), "control");
		context = new ContextImpl();
		
		
		// --- Parameters ---
		try {
			parser.parseControl(null, context);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		try {
			parser.parseControl(source, null);
			Assert.fail("Expected an exception, but none was thrown.");
		} catch (IllegalArgumentException e) {
			// --- Expected exception. Do nothing. ---
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		
		
		// --- Parse ---
		try {
			control = parser.parseControl(source, context);
		} catch (IOException | ParseException e) {
			e.printStackTrace();
			Assert.fail("An unexpected exception was thrown: " + e.getMessage());
			return;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals(0, warnings.size());
		
		Assert.assertNotNull(control);
		Assert.assertTrue(control.getClass().getCanonicalName(), control instanceof BinaryControl);
		binaryControl = (BinaryControl) control;
		Assert.assertNotNull(binaryControl.getPackage());
		Assert.assertEquals("mypackage", binaryControl.getPackage().getName());
		Assert.assertNull(binaryControl.getSource());

		Assert.assertNotNull(binaryControl.getVersion());
		Assert.assertNull(binaryControl.getVersion().getEpoch());
		Assert.assertEquals("1.0.0", binaryControl.getVersion().getUpstreamVersion());
		Assert.assertNull(binaryControl.getVersion().getDebianRevision());
		Assert.assertEquals("1.0.0", binaryControl.getVersion().getText());

		Assert.assertNotNull(binaryControl.getSection());
		Assert.assertEquals("net", binaryControl.getSection().getText());
		Assert.assertNotNull(binaryControl.getPriority());
		Assert.assertEquals("optional", binaryControl.getPriority().getText());
		Assert.assertNotNull(binaryControl.getArchitecture());
		Assert.assertEquals("amd64", binaryControl.getArchitecture().getText());
		Assert.assertNull(binaryControl.getMultiArchitecture());
		Assert.assertNull(binaryControl.getEssential());
		
		dependencies = binaryControl.getDepends();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(1, dependencies.size());
		dependency = dependencies.get(0);
		Assert.assertNotNull(dependency);
		Assert.assertFalse(dependency.isCondition());
		Assert.assertNotNull(dependency.getPackage());
		Assert.assertEquals("oracle-8-jre", dependency.getPackage().getName());
		Assert.assertNotNull(dependency.getRelationOperator());
		Assert.assertEquals(">=", dependency.getRelationOperator().getText());
		Assert.assertNotNull(dependency.getVersion());
		Assert.assertNull(dependency.getVersion().getEpoch());
		Assert.assertEquals("1.8.0", dependency.getVersion().getUpstreamVersion());
		Assert.assertNull(dependency.getVersion().getDebianRevision());
		Assert.assertEquals("1.8.0", dependency.getVersion().getText());
		
		dependencies = binaryControl.getRecommends();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(0, dependencies.size());
		
		dependencies = binaryControl.getSuggests();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(0, dependencies.size());
		
		dependencies = binaryControl.getEnhances();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(0, dependencies.size());

		dependencies = binaryControl.getPreDepends();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(0, dependencies.size());

		dependencies = binaryControl.getBreaks();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(0, dependencies.size());

		dependencies = binaryControl.getConflicts();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(0, dependencies.size());

		dependencies = binaryControl.getProvides();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(0, dependencies.size());

		dependencies = binaryControl.getReplaces();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(0, dependencies.size());

		dependencies = binaryControl.getBuiltUsing();
		Assert.assertNotNull(dependencies);
		Assert.assertEquals(0, dependencies.size());

		Assert.assertNotNull(binaryControl.getInstalledSize());
		Assert.assertEquals(12641280, binaryControl.getInstalledSize().getBytes());
		Assert.assertNotNull(binaryControl.getMaintainer());
		Assert.assertEquals("John Doe", binaryControl.getMaintainer().getName());
		Assert.assertEquals("j.doe@example.com", binaryControl.getMaintainer().getAddress());
		Assert.assertNull(binaryControl.getOriginalMaintainer());
		Assert.assertNull(binaryControl.getModuleAliases());
		Assert.assertNotNull(binaryControl.getDescription());
		Assert.assertEquals(
				"This is my first package.\n" + "I'm somewhat excited.\n" + "\n" + "This will be a great package.\n",
				binaryControl.getDescription().getText());
		Assert.assertNotNull(binaryControl.getHomepage());
		Assert.assertNotNull(binaryControl.getHomepage().getURL());
		Assert.assertEquals("http://www.example.com/", binaryControl.getHomepage().getURL().toString());
	}


}
