/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;

import org.apache.commons.logging.impl.SimpleLog;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;

import net.sourceforge.javadpkg.ChangeLogUrgency;
import net.sourceforge.javadpkg.ChangeLogUrgencyParser;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.Warning;
import net.sourceforge.javadpkg.control.PackageMaintainer;
import net.sourceforge.javadpkg.control.PackageMaintainerParser;
import net.sourceforge.javadpkg.control.PackageName;
import net.sourceforge.javadpkg.control.PackageNameParser;
import net.sourceforge.javadpkg.control.PackageVersion;
import net.sourceforge.javadpkg.control.PackageVersionParser;
import net.sourceforge.javadpkg.control.Size;
import net.sourceforge.javadpkg.control.impl.ArchitectureParserImpl;
import net.sourceforge.javadpkg.control.impl.BinaryControlImpl;
import net.sourceforge.javadpkg.control.impl.DescriptionParserImpl;
import net.sourceforge.javadpkg.control.impl.HomepageParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageDependencyParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageMaintainerParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageNameParserImpl;
import net.sourceforge.javadpkg.control.impl.PackagePriorityParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionRelationOperatorParserImpl;
import net.sourceforge.javadpkg.control.impl.SectionParserImpl;
import net.sourceforge.javadpkg.impl.ChangeLogUrgencyParserImpl;
import net.sourceforge.javadpkg.impl.ContextImpl;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.impl.DataByteArraySource;


/**
 * <p>
 * The abstract superclass of all tests.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 27.12.2015 by Gerrit Hohl
 */
public abstract class AbstractDpkgTest {
	
	
	/**
	 * <p>
	 * Performs some preparations before all tests are performed.
	 * </p>
	 *
	 * @throws Exception
	 *             If an error occurs.
	 */
	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		Properties props;


		props = System.getProperties();
		props.setProperty("org.apache.commons.logging.Log", SimpleLog.class.getCanonicalName());
		props.setProperty("org.apache.commons.logging.simplelog.defaultlog", "trace");
		props.setProperty("org.apache.commons.logging.simplelog.showlogname", "true");
		props.setProperty("org.apache.commons.logging.simplelog.showShortLogname", "false");
		props.setProperty("org.apache.commons.logging.simplelog.showdatetime", "true");
		props.setProperty("org.apache.commons.logging.simplelog.dateTimeFormat", "yyyy-MM-dd HH:mm:ss.SSS");
	}


	/**
	 * <p>
	 * Performs some clean-up after all tests have been performed.
	 * </p>
	 *
	 * @throws Exception
	 *             If an error occurs.
	 */
	@AfterClass
	public static void tearDownAfterClass() throws Exception {
		Properties props;


		props = System.getProperties();
		props.remove("org.apache.commons.logging.Log");
		props.remove("org.apache.commons.logging.simplelog.defaultlog");
		props.remove("org.apache.commons.logging.simplelog.showlogname");
		props.remove("org.apache.commons.logging.simplelog.showShortLogname");
		props.remove("org.apache.commons.logging.simplelog.showdatetime");
		props.remove("org.apache.commons.logging.simplelog.dateTimeFormat");
	}


	/**
	 * <p>
	 * Performs some preparations before each test is performed.
	 * </p>
	 *
	 * @throws Exception
	 *             If an error occurs.
	 */
	@Before
	public void setUp() throws Exception {
		// --- Do nothing. ---
	}


	/**
	 * <p>
	 * Performs some clean-up after each test is performed.
	 * </p>
	 *
	 * @throws Exception
	 *             If an error occurs.
	 */
	@After
	public void tearDown() throws Exception {
		// --- Do nothing. ---
	}


	/**
	 * <p>
	 * Creates a source.
	 * </p>
	 *
	 * @param length
	 *            The length of the data.
	 * @param name
	 *            The name.
	 * @return The source.
	 */
	public static DataSource createSource(int length, String name) {
		DataSource source;
		ByteArrayOutputStream out;
		byte[] buffer;
		int rest, len;
		
		
		Assert.assertTrue(length >= 0);
		Assert.assertNotNull(name);
		
		// --- Create an array with the desired length ---
		out = new ByteArrayOutputStream(length);
		buffer = new byte[65536];
		rest = length;
		do {
			len = rest;
			if (len > buffer.length) {
				len = buffer.length;
			}
			rest -= len;
			if (len > 0) {
				out.write(buffer, 0, len);
			}
		} while (rest > 0);
		
		// --- Create a source based on that array ---
		source = new DataByteArraySource(out.toByteArray(), name, false, true);
		return source;
	}
	
	
	/**
	 * <p>
	 * Creates a source.
	 * </p>
	 *
	 * @param lines
	 *            The lines which will be available through the source.
	 * @param name
	 *            The name.
	 * @return The source.
	 */
	public static DataSource createSource(String lines, String name) {
		DataSource source;
		
		
		Assert.assertNotNull(lines);
		Assert.assertNotNull(name);
		
		source = createSource(Arrays.asList(lines), name);
		return source;
	}


	/**
	 * <p>
	 * Creates a source.
	 * </p>
	 *
	 * @param lines
	 *            The lines which will be available through the source.
	 * @param name
	 *            The name.
	 * @return The source.
	 */
	public static DataSource createSource(List<String> lines, String name) {
		DataSource source;
		ByteArrayOutputStream out;


		Assert.assertNotNull(lines);
		Assert.assertNotNull(name);

		// --- Create an array out of the lines ---
		out = new ByteArrayOutputStream();
		try {
			try (PrintStream ps = new PrintStream(out, false, "UTF-8")) {
				for (String line : lines) {
					ps.println(line);
				}
			}
		} catch (IOException e) {
			e.printStackTrace();
			Assert.fail("Couldn't prepare source |" + name + "|: " + e.getMessage());
			return null;
		}

		// --- Create a source based on that array ---
		source = new DataByteArraySource(out.toByteArray(), name, false, true);
		return source;
	}
	
	
	/**
	 * <p>
	 * Creates a binary control.
	 * </p>
	 *
	 * @param name
	 *            The package name (mandatory).
	 * @param version
	 *            The version (mandatory).
	 * @param section
	 *            The section (recommended).
	 * @param priority
	 *            The priority (recommended).
	 * @param architecture
	 *            The architecture (mandatory).
	 * @param dependencies
	 *            The dependencies.
	 * @param installedSizeInKiloBytes
	 *            The installed size in kilobytes.
	 * @param homepage
	 *            The home-page.
	 * @param maintainer
	 *            The maintainer (mandatory).
	 * @param description
	 *            The description (mandatory).
	 * @return The binary control.
	 */
	public static BinaryControlImpl createBinaryControl(String name, String version, String section, String priority,
			String architecture, String dependencies, Long installedSizeInKiloBytes, String homepage, String maintainer,
			String description) {
		
		Context context;
		BinaryControlImpl control;
		PackageNameParser packageNameParser;
		PackageVersionParser packageVersionParser;
		List<Warning> warnings;


		Assert.assertNotNull(name);
		Assert.assertNotNull(version);
		Assert.assertNotNull(architecture);
		Assert.assertNotNull(maintainer);
		Assert.assertNotNull(description);

		context = new ContextImpl();
		control = new BinaryControlImpl();
		packageNameParser = new PackageNameParserImpl();
		packageVersionParser = new PackageVersionParserImpl();
		try {
			control.setPackage(packageNameParser.parsePackageName(name, context));
			control.setVersion(packageVersionParser.parsePackageVersion(version, context));
			control.setSection(new SectionParserImpl().parseSection(section, context));
			control.setPriority(new PackagePriorityParserImpl().parsePackagePriority(priority, context));
			control.setArchitecture(new ArchitectureParserImpl().parseArchitecture(architecture, context));
			control.setDepends(
					new PackageDependencyParserImpl(packageNameParser, new PackageVersionRelationOperatorParserImpl(),
							packageVersionParser).parsePackageDependencies(dependencies, context));
			if (installedSizeInKiloBytes != null) {
				control.setInstalledSize(Size.getSizeInKiloBytes(installedSizeInKiloBytes.longValue()));
			}
			control.setHomepage(new HomepageParserImpl().parseHomepage(homepage, context));
			control.setMaintainer(new PackageMaintainerParserImpl().parsePackageMaintainer(maintainer, context));
			control.setDescription(new DescriptionParserImpl().parseDescription(description, context));
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't prepare control: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals("Couldn't create binary control: Got " + warnings.size() + " warning(s).", 0, warnings.size());
		return control;
	}


	/**
	 * <p>
	 * Creates a package name from the specified value.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The package name.
	 */
	public static PackageName createPackageName(String value) {
		PackageName name;
		PackageNameParser packageNameParser;
		Context context;
		List<Warning> warnings;


		Assert.assertNotNull(value);

		packageNameParser = new PackageNameParserImpl();
		context = new ContextImpl();
		try {
			name = packageNameParser.parsePackageName(value, context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't parse package name |" + value + "|: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals("Couldn't create package name |" + value + "|: Got " + warnings.size() + " warning(s).", 0,
				warnings.size());
		return name;
	}


	/**
	 * <p>
	 * Creates a package version from the specified value.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The package version.
	 */
	public static PackageVersion createPackageVersion(String value) {
		PackageVersion name;
		PackageVersionParser packageVersionParser;
		Context context;
		List<Warning> warnings;


		Assert.assertNotNull(value);

		packageVersionParser = new PackageVersionParserImpl();
		context = new ContextImpl();
		try {
			name = packageVersionParser.parsePackageVersion(value, context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't parse package version |" + value + "|: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals("Couldn't create package version |" + value + "|: Got " + warnings.size() + " warning(s).", 0,
				warnings.size());
		return name;
	}
	
	
	/**
	 * <p>
	 * Creates a package maintainer from the specified value.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The package maintainer.
	 */
	public static PackageMaintainer createPackageMaintainer(String value) {
		PackageMaintainer maintainer;
		PackageMaintainerParser packageMaintainerParser;
		Context context;
		List<Warning> warnings;


		Assert.assertNotNull(value);

		packageMaintainerParser = new PackageMaintainerParserImpl();
		context = new ContextImpl();
		try {
			maintainer = packageMaintainerParser.parsePackageMaintainer(value, context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't parse package maintainer |" + value + "|: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals("Couldn't create package maintainer |" + value + "|: Got " + warnings.size() + " warning(s).", 0,
				warnings.size());
		return maintainer;
	}
	
	
	/**
	 * <p>
	 * Creates a change log urgency from the specified value.
	 * </p>
	 *
	 * @param value
	 *            The value.
	 * @return The change log urgency.
	 */
	public static ChangeLogUrgency createChangeLogUrgency(String value) {
		ChangeLogUrgency urgency;
		ChangeLogUrgencyParser urgencyParser;
		Context context;
		List<Warning> warnings;


		Assert.assertNotNull(value);

		urgencyParser = new ChangeLogUrgencyParserImpl();
		context = new ContextImpl();
		try {
			urgency = urgencyParser.parseChangeLogUrgency(value, context);
		} catch (ParseException e) {
			e.printStackTrace();
			Assert.fail("Couldn't parse change log urgency |" + value + "|: " + e.getMessage());
			return null;
		}
		warnings = context.getWarnings();
		Assert.assertNotNull(warnings);
		Assert.assertEquals("Couldn't create change log urgency |" + value + "|: Got " + warnings.size() + " warning(s).", 0,
				warnings.size());
		return urgency;
	}


}
