/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.store;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.archivers.tar.TarConstants;

import net.sourceforge.javadpkg.DebianPackageConstants;
import net.sourceforge.javadpkg.io.DataConsumer;
import net.sourceforge.javadpkg.io.DataProducer;
import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.FileMode;
import net.sourceforge.javadpkg.io.FileOwner;
import net.sourceforge.javadpkg.io.Streams;
import net.sourceforge.javadpkg.io.impl.DataDigestConsumer;
import net.sourceforge.javadpkg.io.impl.FileModeImpl;
import net.sourceforge.javadpkg.io.impl.FileOwnerImpl;

/**
 * <p>
 * A node representing a directory or file in the {@link DataStore}.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 26.04.2016 by Gerrit Hohl
 */
public class DataStoreNode implements DebianPackageConstants {
	
	
	/** The source of the file content. */
	private DataSource					source;
	/** The name. */
	private String						name;
	/** The target of the symbolic link. */
	private String						target;
	/** The owner. */
	private FileOwner					owner;
	/** The mode. */
	private FileMode					mode;
	
	/** The child nodes. */
	private Map<String, DataStoreNode>	childNodes;
	/** The parent node. */
	private DataStoreNode				parentNode;
	
	
	/**
	 * <p>
	 * Creates the root node.
	 * </p>
	 */
	public DataStoreNode() {
		super();
		
		this.source = null;
		this.name = "";
		this.target = null;
		this.owner = new FileOwnerImpl(ROOT_GROUP_ID, ROOT_GROUP_NAME, ROOT_USER_ID, ROOT_USER_NAME);
		this.mode = new FileModeImpl(DIRECTORY_MODE);
		this.childNodes = new HashMap<>();
		this.parentNode = null;
	}


	/**
	 * <p>
	 * Creates a node for a directory.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @param owner
	 *            The owner.
	 * @param mode
	 *            The mode.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataStoreNode(String name, FileOwner owner, FileMode mode) {
		super();

		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		
		this.source = null;
		this.name = name;
		this.target = null;
		this.owner = owner;
		this.mode = mode;
		this.childNodes = new HashMap<>();
		this.parentNode = null;
	}


	/**
	 * <p>
	 * Creates a node for a file.
	 * </p>
	 *
	 * @param source
	 *            The source for the content of the file.
	 * @param name
	 *            The name.
	 * @param owner
	 *            The owner.
	 * @param mode
	 *            The mode.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataStoreNode(DataSource source, String name, FileOwner owner, FileMode mode) {
		
		super();
		
		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		
		this.source = source;
		this.name = name;
		this.target = null;
		this.owner = owner;
		this.mode = mode;
		this.childNodes = null;
		this.parentNode = null;
	}
	
	
	/**
	 * <p>
	 * Creates a node for a symbolic link.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @param target
	 *            The target of the symbolic link.
	 * @param owner
	 *            The owner.
	 * @param mode
	 *            The mode.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataStoreNode(String name, String target, FileOwner owner, FileMode mode) {
		super();
		
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (target == null)
			throw new IllegalArgumentException("Argument target is null.");
		if (owner == null)
			throw new IllegalArgumentException("Argument owner is null.");
		if (mode == null)
			throw new IllegalArgumentException("Argument mode is null.");
		
		this.source = null;
		this.name = name;
		this.target = target;
		this.owner = owner;
		this.mode = mode;
		this.childNodes = null;
		this.parentNode = null;
	}
	
	
	/**
	 * <p>
	 * Returns the name.
	 * </p>
	 *
	 * @return The name.
	 */
	public String getName() {
		return this.name;
	}


	/**
	 * <p>
	 * Returns the path.
	 * </p>
	 *
	 * @return The path.
	 */
	public String getPath() {
		StringBuilder sb;
		
		
		sb = new StringBuilder();
		this.addPath(sb);
		return sb.toString();
	}


	/**
	 * <p>
	 * Adds the path to the specified builder.
	 * </p>
	 *
	 * @param builder
	 *            The builder.
	 */
	private void addPath(StringBuilder builder) {
		if (this.parentNode != null) {
			this.parentNode.addPath(builder);
		}
		builder.append(this.name);
		if (this.childNodes != null) {
			builder.append('/');
		}
	}


	/**
	 * <p>
	 * Returns the flag if the node is a directory node.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the node is a directory node,
	 *         <code>false</code> otherwise.
	 * @see #isFile()
	 */
	public boolean isDirectory() {
		return (this.childNodes != null);
	}
	
	
	/**
	 * <p>
	 * Returns the flag if the node is a file node.
	 * </p>
	 * <p>
	 * Symbolic links are also files.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the node is a file node,
	 *         <code>false</code> otherwise.
	 * @see #isDirectory()
	 * @see #isSymbolicLink()
	 */
	public boolean isFile() {
		return ((this.source != null) || (this.target != null));
	}
	
	
	/**
	 * <p>
	 * Returns the flag if the node is a symbolic link node.
	 * </p>
	 *
	 * @return The flag: <code>true</code>, if the node is a symbolic link node,
	 *         <code>false</code> otherwise.
	 * @see #isFile()
	 */
	public boolean isSymbolicLink() {
		return (this.target != null);
	}
	
	
	/**
	 * <p>
	 * Returns the size of the files under this node and its sub-nodes.
	 * </p>
	 *
	 * @return The size in bytes.
	 * @throws IOException
	 *             If the size of a file can't be determined.
	 */
	public long getSize() throws IOException {
		long size = 0;
		
		
		// --- Directory? ---
		if (this.childNodes != null) {
			if (!this.childNodes.isEmpty()) {
				for (DataStoreNode node : this.childNodes.values()) {
					size += node.getSize();
				}
			}
		}
		// --- Otherwise it is a file (if it is not a symbolic link) ---
		else if (this.target == null) {
			size = this.source.getLength();
			if (size < 0)
				throw new IOException(
						"Couldn't determine size for file |" + this.getPath() + "| (source |" + this.source.getName() + "|).");
		}
		return size;
	}
	
	
	/**
	 * <p>
	 * Adds a child node.
	 * </p>
	 *
	 * @param childNode
	 *            The child node.
	 * @throws IllegalArgumentException
	 *             If the child node is <code>null</code>, this node is not a
	 *             directory node, the child node is already added to a node or
	 *             this node already contains a child node with the same name.
	 */
	public void addChildNode(DataStoreNode childNode) {
		if (childNode == null)
			throw new IllegalArgumentException("Argument childNode is null.");
		if (!this.isDirectory())
			throw new IllegalStateException("Can't add child node |" + childNode.getName() + "| because this node |"
					+ this.getPath() + "| is not a directory node.");
		if (childNode.getParentNode() != null)
			throw new IllegalStateException("Can't add child node |" + childNode.getName()
					+ "| because the child node is already added to parent node |" + childNode.getParentNode() + "|.");
		if (this.childNodes.containsKey(childNode.getName()))
			throw new IllegalStateException("Can't add child node |" + childNode.getName() + "| because this node |"
					+ this.getPath() + "| already contains a child node with that name.");
		
		childNode.setParentNode(this);
		this.childNodes.put(childNode.getName(), childNode);
	}


	/**
	 * <p>
	 * Returns the child node with the specified name.
	 * </p>
	 *
	 * @param name
	 *            The name.
	 * @return The node or <code>null</code>, if this node doesn't contain such
	 *         a node.
	 * @throws IllegalArgumentException
	 *             If the name is <code>null</code> or this node is not a
	 *             directory node.
	 */
	public DataStoreNode getChildNodeByName(String name) {
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
		if (this.childNodes == null)
			throw new IllegalArgumentException("Can't look for child node |" + name + "| because this node |" + this.getPath()
					+ "| is not a directory node.");
		
		return this.childNodes.get(name);
	}
	
	
	/**
	 * <p>
	 * Returns the child nodes as a list.
	 * </p>
	 *
	 * @param sorted
	 *            The flag if the child nodes should be sorted.
	 * @return The list.
	 */
	private List<DataStoreNode> getChildNodes(boolean sorted) {
		List<DataStoreNode> nodes;


		// --- Create a copy of the node list ---
		nodes = new ArrayList<>(this.childNodes.values());

		// --- Sort nodes (if requested) ---
		if (sorted) {
			Collections.sort(nodes, new DataStoreNodeComparator());
		}

		return nodes;
	}
	
	
	/**
	 * <p>
	 * Sets the parent node.
	 * </p>
	 *
	 * @param parentNode
	 *            The parent node.
	 */
	private void setParentNode(DataStoreNode parentNode) {
		this.parentNode = parentNode;
	}


	/**
	 * <p>
	 * Returns the parent node.
	 * </p>
	 *
	 * @return The parent node or <code>null</code>, if the node wasn't add to
	 *         any other node.
	 */
	public DataStoreNode getParentNode() {
		return this.parentNode;
	}
	
	
	/**
	 * <p>
	 * Writes the node and its child nodes into the stream.
	 * </p>
	 *
	 * @param out
	 *            The stream.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws IllegalArgumentException
	 *             If the stream is <code>null</code>.
	 */
	public void write(TarArchiveOutputStream out) throws IOException {
		String name;
		TarArchiveEntry entry;
		
		
		if (out == null)
			throw new IllegalArgumentException("Argument out is null.");
		
		// --- Create the TAR entry ---
		name = "." + this.getPath();
		// --- Do we have a directory or file? ---
		if (this.target == null) {
			entry = new TarArchiveEntry(name);
		}
		// --- Otherwise we have a symbolic link ---
		else {
			entry = new TarArchiveEntry(name, TarConstants.LF_SYMLINK);
		}
		entry.setGroupId(this.owner.getGroupId());
		entry.setGroupName(this.owner.getGroupName());
		entry.setUserId(this.owner.getUserId());
		entry.setUserName(this.owner.getUserName());
		entry.setMode(this.mode.getMode());
		// --- Set the length if we have a file ---
		if (this.source != null) {
			entry.setSize(this.source.getLength());
		}
		// --- Set the target if we have a symbolic link ---
		if (this.target != null) {
			entry.setLinkName(this.target);
		}
		out.putArchiveEntry(entry);
		
		// --- Write the contents if we have a file ---
		if (this.source != null) {
			this.writeSource(out);
		}
		out.closeArchiveEntry();
		
		// --- Write the child nodes if we have a directory ---
		if ((this.childNodes != null) && !this.childNodes.isEmpty()) {
			this.writeNodes(out);
		}
	}
	
	
	/**
	 * <p>
	 * Writes the content of the source into the stream.
	 * </p>
	 *
	 * @param out
	 *            The stream.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	private void writeSource(OutputStream out) throws IOException {
		// --- Copy the content from the node source to the TAR entry ---
		try {
			try (InputStream in = this.source.getInputStream()) {
				Streams.copy(in, out);
			}
		} catch (IOException e) {
			throw new IOException("Couldn't write |" + this.getPath() + "| from source |" + this.source.getName()
					+ "| into the stream: " + e.getMessage());
		}

		// --- Reset the source (if supported) ---
		if (this.source.isResettable()) {
			this.source.reset();
		}
	}
	
	
	/**
	 * <p>
	 * Writes the child nodes into the stream.
	 * </p>
	 *
	 * @param out
	 *            The stream.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	private void writeNodes(TarArchiveOutputStream out) throws IOException {
		List<DataStoreNode> nodes;


		// --- Get sorted nodes ---
		nodes = this.getChildNodes(true);

		// --- Writes nodes ---
		for (DataStoreNode node : nodes) {
			node.write(out);
		}
	}


	/**
	 * <p>
	 * Creates the file hashes for all files within the directory and its
	 * sub-directories (if this node represents a directory) or for the file (if
	 * this node represents a file).
	 * </p>
	 * <p>
	 * Symbolic links are ignored. No hash will be created for them.
	 * </p>
	 *
	 * @param digest
	 *            The digest used for creating the hashes.
	 * @return The hashes.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	public List<FileHash> createFileHashes(MessageDigest digest) throws IOException {
		List<FileHash> hashes;
		List<DataStoreNode> nodes;
		
		
		if (digest == null)
			throw new IllegalArgumentException("Argument md is null.");
		
		hashes = new ArrayList<>();
		// --- Create the hash for the content (if the node is a file) ---
		if (this.source != null) {
			hashes.add(this.createFileHash(digest));
		}
		// --- Otherwise process the child nodes (if the node is a directory and we have child nodes) ---
		else if ((this.childNodes != null) && !this.childNodes.isEmpty()) {
			nodes = this.getChildNodes(true);
			for (DataStoreNode node : nodes) {
				hashes.addAll(node.createFileHashes(digest));
			}
		}

		return hashes;
	}


	/**
	 * <p>
	 * Creates the file hash for this node.
	 * </p>
	 *
	 * @param digest
	 *            The digest.
	 * @return The file hash.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	private FileHash createFileHash(MessageDigest digest) throws IOException {
		FileHash fileHash;
		DataProducer producer;
		DataConsumer consumer;
		byte[] hash;
		String path;


		// --- Create hash ---
		producer = Streams.createProducer(this.source);
		consumer = new DataDigestConsumer(digest, "digest");
		digest.reset();
		Streams.transfer(producer, consumer);
		hash = digest.digest();

		// --- Reset the source (if supported) ---
		if (this.source.isResettable()) {
			this.source.reset();
		}
		
		// --- Create file hash ---
		path = this.getPath();
		fileHash = new FileHashImpl(this.name, path, hash);
		return fileHash;
	}


}
