/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.IOException;
import java.io.OutputStream;

import net.sourceforge.javadpkg.io.CloseHandler;


/**
 * <p>
 * An {@link OutputStream} which can't be closed.
 * </p>
 * <p>
 * Calls of the {@link #close()} method don't have any effect on the underlying
 * stream.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 31.12.2015 by Gerrit Hohl
 */
public class UncloseableOutputStream extends OutputStream {


	/** The underlying stream. */
	private OutputStream	out;
	/** The close handler (optional). */
	private CloseHandler	closeHandler;
							
							
	/**
	 * <p>
	 * Creates a stream.
	 * </p>
	 *
	 * @param out
	 *            The underlying stream.
	 * @throws IllegalArgumentException
	 *             If the underlying stream is <code>null</code>.
	 */
	public UncloseableOutputStream(OutputStream out) {
		super();
		
		if (out == null)
			throw new IllegalArgumentException("Argument out is null.");

		this.out = out;
		this.closeHandler = null;
	}
	
	
	/**
	 * <p>
	 * Creates a stream.
	 * </p>
	 *
	 * @param out
	 *            The underlying stream.
	 * @param closeHandler
	 *            The close handler.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public UncloseableOutputStream(OutputStream out, CloseHandler closeHandler) {
		super();
		
		if (out == null)
			throw new IllegalArgumentException("Argument out is null.");
		if (closeHandler == null)
			throw new IllegalArgumentException("Argument closeHandler is null.");

		this.out = out;
		this.closeHandler = closeHandler;
	}


	@Override
	public void write(int b) throws IOException {
		this.out.write(b);
	}


	@Override
	public void write(byte[] b) throws IOException {
		this.out.write(b);
	}


	@Override
	public void write(byte[] b, int off, int len) throws IOException {
		this.out.write(b, off, len);
	}


	@Override
	public void flush() throws IOException {
		this.out.flush();
	}


	@Override
	public void close() throws IOException {
		this.out.flush();
		if (this.closeHandler != null)
			this.closeHandler.closed();
	}


}
