/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2015 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.IOException;
import java.io.OutputStream;

import net.sourceforge.javadpkg.io.DataTarget;

/**
 * <p>
 * A {@link DataTarget} implementation based on a stream.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 31.12.2015 by Gerrit Hohl
 */
public class DataStreamTarget implements DataTarget {


	/** The stream. */
	private OutputStream			out;
	/**
	 * <p>
	 * The public stream which is returned by the {@link #getOutputStream()}
	 * method.
	 * </p>
	 */
	private UncloseableOutputStream	publicOut;
	/** The name of the source. */
	private String					name;
	/** The flag if the {@link #close()} method will close the stream. */
	private boolean					closeable;
									
									
	/**
	 * <p>
	 * Creates a target.
	 * </p>
	 *
	 * @param out
	 *            The stream.
	 * @param name
	 *            The name of the source.
	 * @param closeable
	 *            The flag if the {@link #close()} method will close the stream.
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataStreamTarget(OutputStream out, String name, boolean closeable) {
		super();

		if (out == null)
			throw new IllegalArgumentException("Argument out is null.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");
			
		this.out = out;
		this.publicOut = null;
		this.name = name;
		this.closeable = closeable;
		
		this.createPublicOutputStream();
	}


	@Override
	public String getName() {
		return this.name;
	}
	
	
	/**
	 * <p>
	 * Creates the {@link OutputStream} which is returned by the
	 * {@link #getOutputStream()} method.
	 * </p>
	 */
	private void createPublicOutputStream() {
		this.publicOut = new UncloseableOutputStream(this.out, new DelegateCloseHandler(this));
	}


	@Override
	public OutputStream getOutputStream() throws IOException {
		return this.publicOut;
	}


	@Override
	public void close() throws IOException {
		try {
			if ((this.out != null) && this.closeable) {
				this.out.close();
			}
		} finally {
			this.publicOut = null;
			this.out = null;
		}
	}
	
	
}
