/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;

import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.io.Streams;

/**
 * <p>
 * A {@link DataTarget} implementation based on a file.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 28.04.2016 by Gerrit Hohl
 */
public class DataFileTarget implements DataTarget {


	/** The file. */
	private File					file;
	/** The stream. */
	private OutputStream			out;
	/**
	 * <p>
	 * The public stream which is returned by the {@link #getOutputStream()}
	 * method.
	 * </p>
	 */
	private UncloseableOutputStream	publicOut;
	/** The name of the source. */
	private String					name;
	/** The flag if the stream is already open. */
	private boolean					opened;
									
									
	/**
	 * <p>
	 * Creates a source.
	 * </p>
	 * <p>
	 * The stream on the file will be opened the first time the
	 * {@link #getOutputStream()} method is called.
	 * </p>
	 *
	 * @param file
	 *            The file.
	 * @throws IllegalArgumentException
	 *             If the file is <code>null</code>.
	 */
	public DataFileTarget(File file) {
		super();
		
		if (file == null)
			throw new IllegalArgumentException("Argument file is null.");
			
		this.file = file;
		this.out = null;
		this.publicOut = null;
		this.name = file.getAbsolutePath();
		this.opened = false;
	}
	
	
	@Override
	public String getName() {
		return this.name;
	}


	/**
	 * <p>
	 * Creates the {@link OutputStream} which is returned by the
	 * {@link #getOutputStream()} method.
	 * </p>
	 */
	private void createPublicOutputStream() {
		this.publicOut = new UncloseableOutputStream(this.out, new DelegateCloseHandler(this));
	}


	/**
	 * <p>
	 * Ensures that the stream is opened.
	 * </p>
	 *
	 * @throws IOException
	 *             If an error occurs while opening the stream or if the source
	 *             is already closed.
	 */
	private void ensureOutputStream() throws IOException {
		if (this.out == null) {
			if (!this.opened) {
				try {
					this.out = Streams.createBufferedFileOutputStream(this.file);
				} catch (FileNotFoundException e) {
					throw new IOException("Couldn't open stream on target |" + this.name + "|: " + e.getMessage());
				}
				this.createPublicOutputStream();
			} else
				throw new IOException("The stream of target |" + this.name + "| is already closed.");
		}
	}


	@Override
	public OutputStream getOutputStream() throws IOException {
		this.ensureOutputStream();
		return this.publicOut;
	}
	
	
	@Override
	public void close() throws IOException {
		try {
			if (this.out != null) {
				this.out.close();
			}
		} finally {
			this.publicOut = null;
			this.out = null;
		}
	}
	
}
