/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import net.sourceforge.javadpkg.io.DataSource;
import net.sourceforge.javadpkg.io.Streams;

/**
 * <p>
 * A {@link DataSource} implementation based on a file.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 26.04.2016 by Gerrit Hohl
 */
public class DataFileSource implements DataSource {
	
	
	/** The file. */
	private File					file;
	/** The flag if the {@link #reset()} method will be supported. */
	private boolean					resettable;
	/** The name of the source. */
	private String					name;
	/** The stream. */
	private InputStream				in;
	/**
	 * <p>
	 * The public stream which is returned by the {@link #getInputStream()}
	 * method.
	 * </p>
	 */
	private UncloseableInputStream	publicIn;
	/** The flag if the stream is already open. */
	private boolean					opened;
	
	
	/**
	 * <p>
	 * Creates a source.
	 * </p>
	 * <p>
	 * The stream on the file will be opened the first time the
	 * {@link #getInputStream()} method is called.
	 * </p>
	 *
	 * @param file
	 *            The file.
	 * @param resettable
	 *            The flag if the {@link #reset()} method will be supported.
	 * @throws IllegalArgumentException
	 *             If the file is <code>null</code>.
	 */
	public DataFileSource(File file, boolean resettable) {
		super();

		if (file == null)
			throw new IllegalArgumentException("Argument file is null.");
		
		this.file = file;
		this.name = file.getAbsolutePath();
		this.in = null;
		this.publicIn = null;
		this.opened = false;
		this.resettable = resettable;
	}


	/**
	 * <p>
	 * Creates a source.
	 * </p>
	 * <p>
	 * The stream on the file will be opened the first time the
	 * {@link #getInputStream()} method is called.
	 * </p>
	 * <p>
	 * The {@link #reset()} method will be supported.
	 * </p>
	 *
	 * @param file
	 *            The file.
	 * @throws IllegalArgumentException
	 *             If the file is <code>null</code>.
	 */
	public DataFileSource(File file) {
		this(file, true);
	}


	@Override
	public String getName() {
		return this.name;
	}
	
	
	@Override
	public long getLength() {
		return this.file.length();
	}
	
	
	@Override
	public boolean isResettable() {
		return this.resettable;
	}


	@Override
	public void reset() throws IOException {
		if (!this.resettable)
			throw new IOException("Source |" + this.name + "| doesn't support a reset.");
		try {
			if (this.in != null) {
				this.in.close();
			}
		} finally {
			this.in = null;
			this.publicIn = null;
			this.opened = false;
		}
	}


	/**
	 * <p>
	 * Creates the {@link InputStream} which is returned by the
	 * {@link #getInputStream()} method.
	 * </p>
	 */
	private void createPublicInputStream() {
		this.publicIn = new UncloseableInputStream(this.in, new DelegateCloseHandler(this));
	}
	
	
	/**
	 * <p>
	 * Ensures that the stream is opened.
	 * </p>
	 *
	 * @throws IOException
	 *             If an error occurs while opening the stream or if the source
	 *             is already closed.
	 */
	private void ensureInputStream() throws IOException {
		if (this.in == null) {
			if (!this.opened) {
				try {
					this.in = Streams.createBufferedFileInputStream(this.file);
				} catch (FileNotFoundException e) {
					throw new IOException("Couldn't open stream on source |" + this.name + "|: " + e.getMessage());
				}
				this.createPublicInputStream();
			} else
				throw new IOException("The stream of source |" + this.name + "| is already closed.");
		}
	}


	@Override
	public InputStream getInputStream() throws IOException {
		this.ensureInputStream();
		return this.publicIn;
	}
	
	
	@Override
	public void close() throws IOException {
		try {
			if (this.in != null) {
				this.in.close();
			}
		} finally {
			this.publicIn = null;
			this.in = null;
		}
	}


}
