/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.io.impl;

import java.io.IOException;
import java.security.MessageDigest;

import net.sourceforge.javadpkg.io.DataConsumer;


/**
 * <p>
 * A {@link DataConsumer} which updates a {@link MessageDigest}.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 27.04.2016 by Gerrit Hohl
 */
public class DataDigestConsumer implements DataConsumer {
	
	
	/** The digest. */
	private MessageDigest	digest;
	/** The name of the consumer in the exception (if one is thrown). */
	private String			name;
							
							
	/**
	 * <p>
	 * Creates a consumer.
	 * </p>
	 *
	 * @param digest
	 *            The digest.
	 * @param name
	 *            The name of the consumer in the exception (if one is thrown).
	 * @throws IllegalArgumentException
	 *             If any of the parameters are <code>null</code>.
	 */
	public DataDigestConsumer(MessageDigest digest, String name) {
		super();
		
		if (digest == null)
			throw new IllegalArgumentException("Argument digest is null.");
		if (name == null)
			throw new IllegalArgumentException("Argument name is null.");

		this.digest = digest;
		this.name = name;
	}
	
	
	@Override
	public String getName() {
		return this.name;
	}
	
	
	@Override
	public void consume(byte[] buffer, int length) throws IOException {
		if (buffer == null)
			throw new IllegalArgumentException("Argument buffer is null.");
		if (length < 0)
			throw new IllegalArgumentException("Argument length is negative: " + length);
		if (length > buffer.length)
			throw new IllegalArgumentException("Argument length is greater than the buffer length. Length: " + length
					+ "; buffer length: " + buffer.length);
					
		this.digest.update(buffer, 0, length);
	}
	
}
