/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.Template;
import net.sourceforge.javadpkg.TemplateTypeBuilder;
import net.sourceforge.javadpkg.Templates;
import net.sourceforge.javadpkg.TemplatesBuilder;
import net.sourceforge.javadpkg.TemplatesConstants;
import net.sourceforge.javadpkg.control.Description;
import net.sourceforge.javadpkg.control.DescriptionBuilder;
import net.sourceforge.javadpkg.control.impl.DescriptionBuilderImpl;
import net.sourceforge.javadpkg.field.Field;
import net.sourceforge.javadpkg.field.FieldBuilder;
import net.sourceforge.javadpkg.field.impl.FieldBuilderImpl;
import net.sourceforge.javadpkg.field.impl.FieldImpl;
import net.sourceforge.javadpkg.io.DataTarget;
import net.sourceforge.javadpkg.io.impl.DataStreamTarget;


/**
 * <p>
 * A {@link TemplatesBuilder} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 29.04.2016 by Gerrit Hohl
 */
public class TemplatesBuilderImpl implements TemplatesBuilder, TemplatesConstants {
	
	
	/** The builder for fields. */
	private FieldBuilder		fieldBuilder;
	/** The builder for the template type. */
	private TemplateTypeBuilder	templateTypeBuilder;
	/** The builder for the description. */
	private DescriptionBuilder	descriptionBuilder;


	/**
	 * <p>
	 * Creates a builder.
	 * </p>
	 */
	public TemplatesBuilderImpl() {
		super();
		
		this.fieldBuilder = new FieldBuilderImpl();
		this.templateTypeBuilder = new TemplateTypeBuilderImpl();
		this.descriptionBuilder = new DescriptionBuilderImpl();
	}
	
	
	@Override
	public String buildTemplates(Templates templates, Context context) throws BuildException {
		String result;
		ByteArrayOutputStream out;
		
		
		if (templates == null)
			throw new IllegalArgumentException("Argument templates is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");

		out = new ByteArrayOutputStream();
		try {
			try (DataTarget target = new DataStreamTarget(out, "templates", false)) {
				this.buildTemplates(templates, target, context);
			}
		} catch (IOException e) {
			throw new BuildException("Couldn't build templates: " + e.getMessage(), e);
		}
		try {
			result = new String(out.toByteArray(), UTF_8);
		} catch (UnsupportedEncodingException e) {
			throw new BuildException("Couldn't build templates: " + e.getMessage(), e);
		}
		return result;
	}
	
	
	@Override
	public void buildTemplates(Templates templates, DataTarget target, Context context) throws IOException, BuildException {
		List<Field> fields;
		
		
		if (templates == null)
			throw new IllegalArgumentException("Argument templates is null.");
		if (target == null)
			throw new IllegalArgumentException("Argument target is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
			
		// --- Build the fields from the templates ---
		fields = this.buildFields(templates, context);
		
		// --- Write the fields to the target ---
		this.writeFields(fields, target, context);
	}


	/**
	 * <p>
	 * Builds the fields from the templates.
	 * </p>
	 *
	 * @param templates
	 *            The templates.
	 * @param context
	 *            The context.
	 * @return The fields.
	 * @throws BuildException
	 *             If an error occurs during the building.
	 */
	private List<Field> buildFields(Templates templates, Context context) throws BuildException {
		List<Field> fields;


		fields = new ArrayList<>();
		for (Template template : templates.getTemplates()) {
			// --- Template (name) ---
			fields.add(new FieldImpl(FIELD_TEMPLATE, template.getName()));
			
			// --- Type ---
			try {
				fields.add(new FieldImpl(FIELD_TYPE, this.templateTypeBuilder.buildTemplateType(template.getType(), context)));
			} catch (BuildException e) {
				throw new BuildException("Couldn't build template |" + template.getName() + "|: Couldn't build field |"
						+ FIELD_TYPE + "|: " + e.getMessage(), e);
			}
			
			// --- Default ---
			this.buildDefaultValues(fields, template.getDefaultValue());
			
			// --- Choices ---
			this.buildChoices(fields, template.getChoices());
			
			// --- Description ---
			try {
				this.buildDescriptions(fields, template.getDescriptions(), context);
			} catch (BuildException e) {
				throw new BuildException("Couldn't build template |" + template.getName() + "|: " + e.getMessage(), e);
			}
		}
		return fields;
	}
	
	
	/**
	 * <p>
	 * Builds the fields for the default values.
	 * </p>
	 *
	 * @param fields
	 *            The fields.
	 * @param defaultValues
	 *            The default values.
	 */
	private void buildDefaultValues(List<Field> fields, Map<String, String> defaultValues) {
		String value;
		
		
		value = defaultValues.get(FIELD_DEFAULT);
		if (value != null) {
			fields.add(new FieldImpl(FIELD_DEFAULT, value));
		}

		for (Entry<String, String> entry : defaultValues.entrySet()) {
			if (FIELD_DEFAULT.equals(entry.getKey())) {
				continue;
			}

			fields.add(new FieldImpl(FIELD_DEFAULT_PREFIX + entry.getKey(), entry.getValue()));
		}
	}
	
	
	/**
	 * <p>
	 * Builds the fields for the choices.
	 * </p>
	 *
	 * @param fields
	 *            The fields.
	 * @param choices
	 *            The choices.
	 */
	private void buildChoices(List<Field> fields, Map<String, List<String>> choices) {
		List<String> values;
		
		
		values = choices.get(FIELD_CHOICES);
		if (values != null) {
			fields.add(new FieldImpl(FIELD_CHOICES, this.buildChoices(values)));
		}
		
		for (Entry<String, List<String>> entry : choices.entrySet()) {
			if (FIELD_CHOICES.equals(entry.getKey())) {
				continue;
			}
			
			fields.add(new FieldImpl(FIELD_CHOICES_PREFIX + entry.getKey(), this.buildChoices(entry.getValue())));
		}
	}
	
	
	/**
	 * <p>
	 * Builds a value for the choices.
	 * </p>
	 *
	 * @param choices
	 *            The choices.
	 * @return The value.
	 */
	private String buildChoices(List<String> choices) {
		StringBuilder sb;
		
		
		sb = new StringBuilder();
		for (String choice : choices) {
			if (sb.length() > 0) {
				sb.append(", ");
			}
			sb.append(choice);
		}
		return sb.toString();
	}
	
	
	/**
	 * <p>
	 * Builds the fields for the descriptions.
	 * </p>
	 *
	 * @param fields
	 *            The fields.
	 * @param descriptions
	 *            The descriptions.
	 * @param context
	 *            The context.
	 * @throws BuildException
	 *             If an error occurs during the build.
	 */
	private void buildDescriptions(List<Field> fields, Map<String, Description> descriptions, Context context)
			throws BuildException {
			
		Description description;
		String value, name;
		
		
		description = descriptions.get(FIELD_DESCRIPTION);
		if (description != null) {
			try {
				value = this.descriptionBuilder.buildDescription(description, context);
			} catch (BuildException e) {
				throw new BuildException("Couldn't build description |" + FIELD_DESCRIPTION + "|: " + e.getMessage(), e);
			}
			fields.add(new FieldImpl(FIELD_DESCRIPTION, value));
		}
		
		for (Entry<String, Description> entry : descriptions.entrySet()) {
			if (FIELD_DESCRIPTION.equals(entry.getKey())) {
				continue;
			}
			
			name = FIELD_DESCRIPTION_PREFIX + entry.getKey();
			try {
				value = this.descriptionBuilder.buildDescription(entry.getValue(), context);
			} catch (BuildException e) {
				throw new BuildException("Couldn't build description |" + name + "|: " + e.getMessage(), e);
			}
			fields.add(new FieldImpl(name, value));
		}
	}
	
	
	/**
	 * <p>
	 * Writes the fields into the target.
	 * </p>
	 *
	 * @param fields
	 *            The fields.
	 * @param target
	 *            The target.
	 * @param context
	 *            The context.
	 * @throws IOException
	 *             If an I/O error occurs.
	 * @throws BuildException
	 *             If an error occurs during the building.
	 */
	private void writeFields(List<Field> fields, DataTarget target, Context context) throws IOException, BuildException {
		try {
			this.fieldBuilder.buildFields(fields, target, context);
		} catch (IOException e) {
			throw new IOException("Couldn't write fields of templates: " + e.getMessage(), e);
		} catch (BuildException e) {
			throw new BuildException("Couldn't write fields of templates: " + e.getMessage(), e);
		}
	}


}
