/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.List;

import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.GlobalConstants;
import net.sourceforge.javadpkg.ParseException;
import net.sourceforge.javadpkg.SharedLibraries;
import net.sourceforge.javadpkg.SharedLibrariesParser;
import net.sourceforge.javadpkg.control.PackageDependency;
import net.sourceforge.javadpkg.control.PackageDependencyParser;
import net.sourceforge.javadpkg.control.impl.PackageDependencyParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageNameParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionParserImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionRelationOperatorParserImpl;
import net.sourceforge.javadpkg.io.DataSource;


/**
 * <p>
 * A {@link SharedLibrariesParser} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.01.2016 by Gerrit Hohl
 */
public class SharedLibrariesParserImpl implements SharedLibrariesParser, GlobalConstants {


	/** The parser for the package dependencies. */
	private PackageDependencyParser packageDependencyParser;


	/**
	 * <p>
	 * Creates a parser.
	 * </p>
	 */
	public SharedLibrariesParserImpl() {
		super();

		this.packageDependencyParser = new PackageDependencyParserImpl(new PackageNameParserImpl(),
				new PackageVersionRelationOperatorParserImpl(), new PackageVersionParserImpl());
	}
	
	
	@Override
	public SharedLibraries parseSharedLibraries(DataSource source, Context context) throws IOException, ParseException {
		SharedLibrariesImpl sharedLibraries;
		String line, type, libraryName, version;
		String[] parts;
		List<PackageDependency> dependencies;
		int index;
		
		
		if (source == null)
			throw new IllegalArgumentException("Argument source is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
			
		sharedLibraries = new SharedLibrariesImpl();
		try {
			try (BufferedReader reader = new BufferedReader(new InputStreamReader(source.getInputStream(), UTF_8_CHARSET))) {
				while ((line = reader.readLine()) != null) {
					parts = line.split("[ \\t]+", 3);
					if (parts.length != 3)
						throw new ParseException("Couldn't split line |" + line + "| of shared libraries from source |"
								+ source.getName() + "| into 3 parts. Got " + parts.length);
					if (parts[0].endsWith(":")) {
						parts = line.split("[ \\t]+", 4);
						type = parts[0].substring(0, parts[0].length() - 1);
						index = 1;
					} else {
						type = null;
						index = 0;
					}

					libraryName = parts[index++];
					// ROADMAP Maybe the version should parsed also.
					version = parts[index++];
					try {
						dependencies = this.packageDependencyParser.parsePackageDependencies(parts[index], context);
					} catch (ParseException e) {
						throw new ParseException("Couldn't parse the package dependencies of line |" + line
								+ "| of shared libraries from source |" + source.getName() + "|: " + e.getMessage(), e);
					}
					
					sharedLibraries.addSharedLibrary(type, libraryName, version, dependencies);
				}
			}
		} catch (IOException e) {
			throw new IOException(
					"Couldn't read the shared libraries from source |" + source.getName() + "|: " + e.getMessage());
		}
		return sharedLibraries;
	}


}
