/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.util.ArrayList;
import java.util.List;

import net.sourceforge.javadpkg.SharedLibraries;
import net.sourceforge.javadpkg.control.PackageDependency;


/**
 * <p>
 * A {@link SharedLibraries} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 10.01.2016 by Gerrit Hohl
 */
public class SharedLibrariesImpl implements SharedLibraries {
	
	
	/** The shared libraries. */
	private List<SharedLibrary> sharedLibraries;
	
	
	/**
	 * <p>
	 * Creates shared libraries.
	 * </p>
	 */
	public SharedLibrariesImpl() {
		super();
		
		this.sharedLibraries = new ArrayList<>();
	}
	
	
	/**
	 * <p>
	 * Adds a shared library.
	 * </p>
	 *
	 * @param type
	 *            The type (optional).
	 * @param libraryName
	 *            The name of the library.
	 * @param version
	 *            The version.
	 * @param dependencies
	 *            The dependencies.
	 * @throws IllegalArgumentException
	 *             If any of the non-optional parameters are <code>null</code>.
	 */
	public void addSharedLibrary(String type, String libraryName, String version, List<PackageDependency> dependencies) {
		SharedLibrary sharedLibrary;


		if (libraryName == null)
			throw new IllegalArgumentException("Argument libraryName is null.");
		if (version == null)
			throw new IllegalArgumentException("Argument version is null.");
		if (dependencies == null)
			throw new IllegalArgumentException("Argument dependencies is null.");
			
		sharedLibrary = new SharedLibrary(type, libraryName, version, dependencies);
		this.sharedLibraries.add(sharedLibrary);
	}
	
	
	@Override
	public String getText() {
		StringBuilder sb;
		
		
		sb = new StringBuilder();
		for (SharedLibrary lib : this.sharedLibraries) {
			if (sb.length() > 0) {
				sb.append('\n');
			}
			if (lib.getType() != null) {
				sb.append(lib.getType());
				sb.append(':');
			}
			sb.append(' ');
			sb.append(lib.getLibraryName());
			sb.append(' ');
			sb.append(lib.getVersion());
			sb.append(' ');
			// TODO Implement getText() method (need a PackageDependencyTransformer).
		}
		return sb.toString();
	}


	/* **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 * **********************************************************************
	 */


	/**
	 * <p>
	 * A shared library.
	 * </p>
	 *
	 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
	 * @version <b>1.0</b>, 10.01.2016 by Gerrit Hohl
	 */
	private class SharedLibrary {


		/** The type (optional). */
		private String					type;
		/** The name of the library. */
		private String					libraryName;
		/** The version. */
		private String					version;
		/** The dependencies. */
		private List<PackageDependency>	dependencies;
										
										
		/**
		 * <p>
		 * Creates a shared library.
		 * </p>
		 *
		 * @param type
		 *            The type (optional).
		 * @param libraryName
		 *            The name of the library.
		 * @param version
		 *            The version.
		 * @param dependencies
		 *            The dependencies.
		 */
		public SharedLibrary(String type, String libraryName, String version, List<PackageDependency> dependencies) {
			super();

			this.type = type;
			this.libraryName = libraryName;
			this.version = version;
			this.dependencies = new ArrayList<>(dependencies);
		}


		/**
		 * <p>
		 * Returns the type.
		 * </p>
		 *
		 * @return The type or <code>null</code>, if no type is specified.
		 */
		public String getType() {
			return this.type;
		}


		/**
		 * <p>
		 * Returns the name of the library.
		 * </p>
		 *
		 * @return The name.
		 */
		public String getLibraryName() {
			return this.libraryName;
		}


		/**
		 * <p>
		 * Returns the version.
		 * </p>
		 *
		 * @return The version.
		 */
		public String getVersion() {
			return this.version;
		}


		/**
		 * <p>
		 * Returns the dependencies.
		 * </p>
		 *
		 * @return The dependencies.
		 */
		public List<PackageDependency> getDependencies() {
			return (new ArrayList<>(this.dependencies));
		}


	}
	
	
}
