/*
 * dpkg - Debian Package library and the Debian Package Maven plugin
 * (c) Copyright 2016 Gerrit Hohl
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.sourceforge.javadpkg.impl;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.List;

import net.sourceforge.javadpkg.BuildException;
import net.sourceforge.javadpkg.ChangeLog;
import net.sourceforge.javadpkg.ChangeLogBuilder;
import net.sourceforge.javadpkg.ChangeLogConstants;
import net.sourceforge.javadpkg.ChangeLogUrgencyBuilder;
import net.sourceforge.javadpkg.ChangeLogVersionEntry;
import net.sourceforge.javadpkg.ChangeLogVersionEntryDetail;
import net.sourceforge.javadpkg.Context;
import net.sourceforge.javadpkg.control.PackageMaintainerBuilder;
import net.sourceforge.javadpkg.control.PackageNameBuilder;
import net.sourceforge.javadpkg.control.PackageVersionBuilder;
import net.sourceforge.javadpkg.control.impl.PackageMaintainerBuilderImpl;
import net.sourceforge.javadpkg.control.impl.PackageNameBuilderImpl;
import net.sourceforge.javadpkg.control.impl.PackageVersionBuilderImpl;
import net.sourceforge.javadpkg.io.DataTarget;


/**
 * <p>
 * A {@link ChangeLogBuilder} implementation.
 * </p>
 *
 * @author Gerrit Hohl (gerrit-hohl@users.sourceforge.net)
 * @version <b>1.0</b>, 04.05.2016 by Gerrit Hohl
 */
public class ChangeLogBuilderImpl implements ChangeLogBuilder, ChangeLogConstants {
	
	
	/** The newline. */
	private static final String			NEWLINE	= "\n";
	
	
	/** The builder for the package name. */
	private PackageNameBuilder			packageNameBuilder;
	/** The builder for the version. */
	private PackageVersionBuilder		packageVersionBuilder;
	/** The builder for the urgency. */
	private ChangeLogUrgencyBuilder		changeLogUrgencyBuilder;
	/** The builder for the maintainer. */
	private PackageMaintainerBuilder	packageMaintainerBuilder;


	/**
	 * <p>
	 * Creates a builder.
	 * </p>
	 */
	public ChangeLogBuilderImpl() {
		super();
		
		this.packageNameBuilder = new PackageNameBuilderImpl();
		this.packageVersionBuilder = new PackageVersionBuilderImpl();
		this.changeLogUrgencyBuilder = new ChangeLogUrgencyBuilderImpl();
		this.packageMaintainerBuilder = new PackageMaintainerBuilderImpl();
	}


	@Override
	public void buildChangeLog(ChangeLog changeLog, DataTarget target, Context context) throws IOException, BuildException {
		boolean firstEntry = true;
		String value, version;
		List<String> values;
		List<ChangeLogVersionEntryDetail> details;
		
		
		if (changeLog == null)
			throw new IllegalArgumentException("Argument changeLog is null.");
		if (target == null)
			throw new IllegalArgumentException("Argument target is null.");
		if (context == null)
			throw new IllegalArgumentException("Argument context is null.");
		
		try {
			try (BufferedWriter out = new BufferedWriter(new OutputStreamWriter(target.getOutputStream(), UTF_8_CHARSET))) {
				for (ChangeLogVersionEntry entry : changeLog.getEntries()) {
					if (firstEntry) {
						firstEntry = false;
					} else {
						out.write(NEWLINE);
					}

					// --- Package name ---
					try {
						value = this.packageNameBuilder.buildPackageName(entry.getPackage(), context);
					} catch (BuildException e) {
						throw new BuildException("Couldn't build change log |" + target.getName()
								+ "| because package name couldn't be build: " + e.getMessage(), e);
					}
					out.write(value);
					out.write(" (");

					// --- Version ---
					try {
						version = this.packageVersionBuilder.buildPackageVersion(entry.getVersion(), context);
					} catch (BuildException e) {
						throw new BuildException("Couldn't build change log |" + target.getName()
								+ "| because version couldn't be build: " + e.getMessage(), e);
					}
					out.write(version);
					out.write(")");

					// --- Distributions ---
					values = entry.getDistributions();
					if (values.isEmpty())
						throw new IllegalArgumentException("Couldn't build change log |" + target.getName()
								+ "| because no distribution was defined for version |" + version + "|.");
					for (String v : values) {
						out.write(" ");
						out.write(v);
					}
					out.write("; urgency=");
					
					// --- Urgency ---
					try {
						value = this.changeLogUrgencyBuilder.buildChangeLogUrgency(entry.getUrgency(), context);
					} catch (BuildException e) {
						throw new BuildException("Couldn't build change log |" + target.getName()
								+ "| because urgency couldn't be build for version |" + version + "|: " + e.getMessage(), e);
					}
					out.write(value);
					out.write(NEWLINE);

					// --- Details ---
					details = entry.getDetails();
					if (details.isEmpty())
						throw new BuildException("Couldn't build change log |" + target.getName()
								+ "| because no details are defined for version |" + version + "|.");
					for (ChangeLogVersionEntryDetail detail : details) {
						out.write(NEWLINE);
						this.buildDetail(out, detail);
					}
					out.write(NEWLINE);
					
					// --- Maintainer ---
					try {
						value = this.packageMaintainerBuilder.buildPackageMaintainer(entry.getMaintainer(), context);
					} catch (BuildException e) {
						throw new BuildException("Couldn't build change log |" + target.getName()
								+ "| because maintainer couldn't be build for version |" + version + "|: " + e.getMessage(), e);
					}
					out.write(TERMINATION_PREFIX);
					out.write(value);
					out.write(MAINTAINER_TIMESTAMP_SEPARATOR);
					
					// --- Timestamp ---
					value = TIMESTAMP_FORMAT.format(entry.getDate());
					out.write(value);
					out.write(NEWLINE);
				}
			}
		} catch (IOException e) {
			throw new IOException("Couldn't write change log |" + target.getName() + "|: " + e.getMessage(), e);
		}
	}


	/**
	 * <p>
	 * Builds a detail into the writer.
	 * </p>
	 *
	 * @param out
	 *            The writer.
	 * @param detail
	 *            The detail.
	 * @throws IOException
	 *             If an I/O error occurs.
	 */
	private void buildDetail(Writer out, ChangeLogVersionEntryDetail detail) throws IOException {
		String text;
		String[] lines;
		boolean firstLine = true;
		
		
		text = detail.getText();
		lines = text.split("\n", -1);
		for (String line : lines) {
			if (firstLine) {
				out.write(DETAIL_START_PREFIX);
				firstLine = false;
			} else {
				out.write(NEWLINE);
				out.write(DETAIL_LINE_PREFIX);
			}

			if (line.isEmpty()) {
				out.write(".");
			} else {
				out.write(line);
			}
		}
		out.write(NEWLINE);
	}
	
	
	@Override
	public void buildChangeLogHtml(ChangeLog changeLog, DataTarget target, Context context) throws IOException, BuildException {
		// TODO Implement method.
		throw new BuildException("Not yet implemented");
	}


}
